<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    exclude-result-prefixes="xs"
    version="3.0">
    
    <!-- produce components list with status: 'new', 'modified', 'unchanged', or 'unknown' 
    
         Should work even if folder depth levels are uneven between two versions of schemas
         
         
         1. identify if simpleType or complexType was modified
         2. identify if components were new or deleted
            if so, identify if moved from IP namespace to common or vice versa
         3. identify if pre-existing elements were modified
            if not, check if its type was modified
-->
    <xsl:param name="previousSchemaInputDir"/>
    
    <xsl:param name="newSchemaComponents"/>
    <xsl:param name="preexistingSchemaComponents"/>
    <xsl:param name="deletedComponents"/>
    <xsl:param name="previousSchemaVersion"/>
    
    <xsl:variable name="previousSchemaCollection" select="collection(concat('file:///',$previousSchemaInputDir,'?select=*.xsd&amp;recurse=yes'))[not(contains(base-uri(.),'/ExternalStandards/'))]"/>
    
    <xsl:key name="schema-by-name-and-namespace" match="*:schema/*[@name]" use="concat(@name,tokenize(parent::*:schema/@targetNamespace,'[/:]')[last()])"/><!-- e.g. 'TrademarkDocumentType,Trademark' -->
    <xsl:key name="existing-component-by-type" match="component" use="concat(@name,@namespace)"/>
    
    <xsl:template match="/" mode="component-status">
        <xsl:variable name="modified-xsd-types">
            <xsl:merge>
                <xsl:merge-source name="preexistingSchemaTypes" for-each-source="tokenize($preexistingSchemaComponents, ';')" select="snapshot(/*:schema/*[not(local-name() = ('element','attribute'))][@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-source name="previousSchemaTypes" select="snapshot($previousSchemaCollection/*:schema/*[not(local-name() = ('element','attribute'))][@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-source name="deletedSchemaFiles" for-each-source="tokenize($deletedComponents,';')" select="snapshot(/*:schema/*[not(local-name() = ('element','attribute'))][@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-action>
                    <xsl:merge>
                        <xsl:merge-source name="previousSchemaTypeByNamespace" select="current-merge-group('previousSchemaTypes')" sort-before-merge="true">
                            <xsl:merge-key select="@name"/>
                            <xsl:merge-key select="parent::node()/@targetNamespace/tokenize(., '[/:]')[last()]"/>
                        </xsl:merge-source>
                        <xsl:merge-source name="preexistingSchemaTypeByNamespace" select="current-merge-group('preexistingSchemaTypes')" sort-before-merge="true">
                            <xsl:merge-key select="@name"/>
                            <xsl:merge-key select="parent::node()/@targetNamespace/tokenize(., '[/:]')[last()]"/>
                        </xsl:merge-source>
                        <xsl:merge-action>

                            <xsl:variable name="componentName" select="current-merge-key()[1]"/>
                            <xsl:variable name="namespace" select="current-merge-key()[2]"/>

                            <xsl:choose>
                                <!-- when flattened, check for new/deleted -->
                                <xsl:when test="exists(current-merge-group('previousSchemaTypeByNamespace')) and deep-equal(current-merge-group('previousSchemaTypeByNamespace')[1], current-merge-group('preexistingSchemaTypeByNamespace')[1]) = false()">
                                    <!-- compare only at the component level; this works in conjunction with saxon setting to strip whitespace only nodes (see ant build) -->
                                    <xsl:variable name="only-modified-annotation" select="deep-equal(current-merge-group('previousSchemaTypeByNamespace')[1]/descendant::*[not(ancestor-or-self::*:annotation[1])], current-merge-group('preexistingSchemaTypeByNamespace')[1]/descendant::*[not(ancestor-or-self::*:annotation[1])]) and deep-equal(current-merge-group('previousSchemaTypeByNamespace')[1]/descendant-or-self::*[not(ancestor-or-self::*:annotation[1])]/@*, current-merge-group('preexistingSchemaTypeByNamespace')[1]/descendant-or-self::*[not(ancestor-or-self::*:annotation[1])]/@*)"/>
                                    <component uri="{base-uri(.)}" status="modified" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" only-modified-annotation="{$only-modified-annotation}" type="{local-name()}"/>
                                    <!-- in the future, consider: ignore order of xsd:enumeration value and/or xsd:choice?? -->
                                </xsl:when>
                                <xsl:when test="exists(current-merge-group('previousSchemaTypeByNamespace')) and deep-equal(current-merge-group('previousSchemaTypeByNamespace')[1], current-merge-group('preexistingSchemaTypeByNamespace')[1]) = true()">
                                    <component uri="{base-uri(.)}" status="unchanged" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" type="{local-name()}"/>
                                </xsl:when>
                                <xsl:when test="empty(current-merge-group('previousSchemaTypeByNamespace'))">
                                    <xsl:message>new in flattened schema: <xsl:value-of select="current-merge-key()[1]"/></xsl:message>
                                    <xsl:variable name="moved-to-common" select="if (matches(current-merge-key()[2],'common','i')) then (exists(current-merge-group('deletedSchemaFiles')[not(matches(parent::node()/@targetNamespace/tokenize(., '[/:]')[last()],'common','i'))])) else (false())"/>
                                    <component uri="{base-uri(.)}" status="new" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" type="{local-name()}" moved-namespaces="{$moved-to-common}"/>
                                </xsl:when>
                            </xsl:choose>
                        </xsl:merge-action>
                    </xsl:merge></xsl:merge-action>
            </xsl:merge>
        </xsl:variable>
        
        <components>
            <xsl:copy-of select="$modified-xsd-types"/>
            <xsl:merge>
                <xsl:merge-source name="newSchemaFiles" for-each-source="tokenize($newSchemaComponents, ';')" select="snapshot(/*:schema/*[@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-source name="deletedSchemaFiles" for-each-source="tokenize($deletedComponents,';')" select="snapshot(/*:schema/*[@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-source name="preexistingSchemaFiles" for-each-source="tokenize($preexistingSchemaComponents, ';')" select="snapshot(/*:schema/*[@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>
                </xsl:merge-source>
                <xsl:merge-source name="previousSchemaFiles" select="snapshot($previousSchemaCollection/*:schema/*[@name])" sort-before-merge="true">
                    <xsl:merge-key select="@name"/>

                </xsl:merge-source>
                <xsl:merge-action>
                    <xsl:for-each-group select="current-merge-group()" group-by="parent::node()/@targetNamespace/tokenize(., '[/:]')[last()]">
                        <!-- identify which components which are new/deleted were moved to/from common namespace -->
                        <xsl:choose>
                            <xsl:when test="exists(current-merge-group('newSchemaFiles')[parent::node()/@targetNamespace/tokenize(., '[/:]')[last()] eq current-grouping-key()]) and empty(current-merge-group('previousSchemaFiles')[parent::node()/@targetNamespace/tokenize(., '[/:]')[last()] eq current-grouping-key()])">
                                <xsl:variable name="moved-to-common" select="if (matches(current-grouping-key(),'common','i')) then (exists(current-merge-group('deletedSchemaFiles')[not(matches(parent::node()/@targetNamespace/tokenize(., '[/:]')[last()],'common','i'))])) else (false())"/>
                                <component uri="{base-uri(.)}" status="new" namespace="{current-grouping-key()}" name="{current-merge-key()[1]}" type="{local-name()}" moved-namespaces="{$moved-to-common}"/>
                            </xsl:when>
                            <xsl:when test="exists(current-merge-group('deletedSchemaFiles')[parent::node()/@targetNamespace/tokenize(., '[/:]')[last()] eq current-grouping-key()]) and empty(current-merge-group('preexistingSchemaFiles')[parent::node()/@targetNamespace/tokenize(., '[/:]')[last()] eq current-grouping-key()])">
                                <xsl:variable name="moved-to-common" select="if (not(matches(current-grouping-key(),'common','i'))) then (exists(current-merge-group('newSchemaFiles')[(matches(parent::node()/@targetNamespace/tokenize(., '[/:]')[last()],'common','i'))])) else (false())"/>
                                <component uri="{base-uri(.)}" status="deleted" namespace="{current-grouping-key()}" name="{current-merge-key()[1]}" type="{local-name()}" moved-namespaces="{$moved-to-common}"/>
                            </xsl:when>
                            <xsl:when test="exists(current-merge-group('preexistingSchemaFiles'))"/>
                            <xsl:when test="exists(current-merge-group('previousSchemaFiles')[parent::node()/@targetNamespace/tokenize(., '[/:]')[last()] eq current-grouping-key()]) ">
                                <xsl:message>deleted in flattened schema: <xsl:value-of select="current-merge-key()[1]"/></xsl:message>
                                <xsl:variable name="moved-to-common" select="if (not(matches(current-grouping-key(),'common','i'))) then (exists(current-merge-group('newSchemaFiles')[(matches(parent::node()/@targetNamespace/tokenize(., '[/:]')[last()],'common','i'))])) else (false())"/>
                                <component uri="{base-uri(.)}" status="deleted" namespace="{current-grouping-key()}" name="{current-merge-key()[1]}" type="{local-name()}" moved-namespaces="{$moved-to-common}"/>

                            </xsl:when>
                        </xsl:choose>
                    </xsl:for-each-group>

                    <!-- preexisting components: determine if they are modified, if unchanged element/attribute, check the simple/complexType's status  -->
                    <xsl:merge>
                        <xsl:merge-source name="preexistingSchemaComponents" select="snapshot(current-merge-group('preexistingSchemaFiles')[(local-name() = ('element','attribute'))][@name])" sort-before-merge="true">
                            <xsl:merge-key select="@name"/>
                            <xsl:merge-key select="parent::node()/@targetNamespace/tokenize(., '[/:]')[last()]"/>
                        </xsl:merge-source>
                        <xsl:merge-source name="previousSchemaComponents" select="snapshot(current-merge-group('previousSchemaFiles')[(local-name() = ('element','attribute'))][@name])" sort-before-merge="true">
                            <xsl:merge-key select="@name"/>
                            <xsl:merge-key select="parent::node()/@targetNamespace/tokenize(., '[/:]')[last()]"/>
                        </xsl:merge-source>
                        <xsl:merge-action>
                            
                            <xsl:variable name="componentName" select="current-merge-key()[1]"/>
                            <xsl:variable name="namespace" select="current-merge-key()[2]"/>
                            <xsl:variable name="typeLocalName" select="current-merge-group('preexistingSchemaComponents')/local-name-from-QName(resolve-QName(@type,.))"/>
                            <xsl:variable name="typeNamespace" select="tokenize(current-merge-group('preexistingSchemaComponents')/namespace-uri-from-QName(resolve-QName(@type,.)), '[/:]')[last()]"/>
                            <xsl:variable name="typeStatus" select="$modified-xsd-types/key('existing-component-by-type',concat($typeLocalName,$typeNamespace))/@status"/>
                            <xsl:choose>
                                <!-- when flattened, check for new/deleted -->
                                <xsl:when test="exists(current-merge-group('previousSchemaComponents')) and deep-equal(current-merge-group('previousSchemaComponents')[1], current-merge-group('preexistingSchemaComponents')[1]) = false()">
                                    <!-- compare only at the component level; this works in conjunction with saxon setting to strip whitespace only nodes (see ant build) -->
                                    <xsl:variable name="only-modified-annotation" select="deep-equal(current-merge-group('previousSchemaComponents')[1]/descendant::*[not(ancestor-or-self::*:annotation[1])], current-merge-group('preexistingSchemaComponents')[1]/descendant::*[not(ancestor-or-self::*:annotation[1])]) and deep-equal(current-merge-group('previousSchemaComponents')[1]/descendant-or-self::*[not(ancestor-or-self::*:annotation[1])]/@*, current-merge-group('preexistingSchemaComponents')[1]/descendant-or-self::*[not(ancestor-or-self::*:annotation[1])]/@*)"/>

                                    <!-- in the future, consider: ignore order of xsd:enumeration value and/or xsd:choice?? -->

                                    <xsl:choose>
                                        <xsl:when test="$typeStatus = 'modified'">
                                            <xsl:message>no change to contents of <xsl:value-of select="local-name()"/> model but type was modified</xsl:message>
                                            <component uri="{base-uri(.)}" status="modified" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" only-modified-annotation="{$typeStatus/parent::node()/@only-modified-annotation}" type="{local-name()}"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <component uri="{base-uri(.)}" status="modified" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" only-modified-annotation="{$only-modified-annotation}" type="{local-name()}"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:when>
                                <xsl:when test="exists(current-merge-group('previousSchemaComponents')) and deep-equal(current-merge-group('previousSchemaComponents')[1], current-merge-group('preexistingSchemaComponents')[1]) = true()">
                                    <xsl:choose>
                                        <xsl:when test="$typeStatus = 'modified'">
                                            <xsl:message><xsl:value-of select="current-merge-key()[1]"/> (<xsl:value-of select="current-merge-key()[2]"/>) deep-equal but type was modified</xsl:message>
                                            <component uri="{base-uri(.)}" status="modified" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" only-modified-annotation="{$typeStatus/parent::node()/@only-modified-annotation}" type="{local-name()}"/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <component uri="{base-uri(.)}" status="unchanged" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" type="{local-name()}"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:when>
                                <xsl:when test="empty(current-merge-group('previousSchemaComponents'))">
                                    <xsl:message>new in flattened schema: <xsl:value-of select="current-merge-key()[1]"/></xsl:message>
                                    <xsl:variable name="moved-to-common" select="if (matches(current-merge-key()[2],'common','i')) then (exists(current-merge-group('deletedSchemaFiles')[not(matches(parent::node()/@targetNamespace/tokenize(., '[/:]')[last()],'common','i'))])) else (false())"/>
                                    <component uri="{base-uri(.)}" status="new" namespace="{current-merge-key()[2]}" name="{current-merge-key()[1]}" type="{local-name()}" moved-namespaces="{$moved-to-common}"/>
                                </xsl:when>
                            </xsl:choose>
                        </xsl:merge-action>
                    </xsl:merge>
                </xsl:merge-action>
            </xsl:merge>
        </components>
    </xsl:template>
    
</xsl:stylesheet>