<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    exclude-result-prefixes="xs"
    version="2.0">
    
    <!-- extract Approved Acronyms and Abbreviations into dev/AcronymsAndAbbreviations.xml
         extract General and Standard Design rules into dev/st96rules.xml
    -->
    <xsl:param name="drcDocFileName" select="'03-96-i.pdf'"/>
    
    
    <xsl:template match="/">
        <xsl:result-document href="{resolve-uri('../st96rules.xml')}" method="xml" indent="yes" encoding="UTF-8">
            <st96rules>
                <xsl:attribute name="drcDocFileName">
                    <xsl:value-of select="$drcDocFileName"/>
                </xsl:attribute>
                <xsl:attribute name="drcDocLastModifiedDate">
                    <xsl:variable name="modified-date">
                        <xsl:apply-templates select="/*:document/*:body/*:p[matches(., '^Revision approved.*')]"/>    
                    </xsl:variable>
                    
                    <xsl:analyze-string select="replace($modified-date, '^.*Revision approved .* on ', '')" regex="(((\d+)\s+(January|February|March|April|May|June|July|August|September|October|November|December))|((January|February|March|April|May|June|July|August|September|October|November|December)\s+(\d+))),?\s+(\d\d\d\d)">
                        <xsl:matching-substring>
                            <!-- convert from format-date(xs:date($pubdate),'[MNn] [D], [Y]') to YYYY-MM-DD -->
                            <xsl:value-of select="concat(regex-group(8), '-', format-number(string-length(substring-before('JanFebMarAprMayJunJulAugSepOctNovDec', substring(concat(regex-group(4), regex-group(6)), 1, 3))) div 3 + 1, '00'), '-', format-number(number(concat(regex-group(3), regex-group(7))), '00'))"/>
                        </xsl:matching-substring>
                        <xsl:non-matching-substring>
                            <xsl:analyze-string select="current()" regex="\d\d\d\d-[0-1]\d-[0-3]\d">
                                <xsl:matching-substring>
                                    <!-- convert from format-date(xs:date($pubdate),'[MNn] [D], [Y]') to YYYY-MM-DD -->
                                        <xsl:value-of select="regex-group(0)"/>
                                </xsl:matching-substring>
                                <xsl:non-matching-substring>
                                    <xsl:value-of select="current()"/>
                                </xsl:non-matching-substring>
                            </xsl:analyze-string>
                        </xsl:non-matching-substring>
                    </xsl:analyze-string>
                </xsl:attribute>
                <xsl:attribute name="drcVersionNum">
                    <xsl:variable name="version">
                        <xsl:apply-templates select="/*:document/*:body/*:p[matches(., '^\s*Version \d+\.\d+\s*$')]"/>    
                    </xsl:variable>
                    <xsl:value-of select="replace($version,'^\s*Version ','')"/>
                </xsl:attribute>
                <xsl:message>Extracting updated ST.96 rules</xsl:message>
                <xsl:apply-templates select="/*:document/*:body/*:tbl[*:tr[1]/*:tc[1][contains(., 'Rule ID')]]" mode="rules"/>
            </st96rules>
        </xsl:result-document>
        
        <xsl:result-document href="{resolve-uri('../AcronymsAndAbbreviations.xml')}" method="xml" indent="yes" encoding="UTF-8">
            <AAs>
                <xsl:message>Extracting updated Acronyms and Abbreviations</xsl:message>
                <xsl:apply-templates select="/*:document/*:body/*:tbl[preceding-sibling::*:p[matches(., '^\s*APPENDIX C - LIST OF ACRONYMS AND ABBREVIATIONS\s*$', 'i')][1]/following-sibling::*:tbl[1] is .]" mode="abbreviation"/>
            </AAs>
        </xsl:result-document>
        
    </xsl:template>
    
    <xsl:template match="*:tbl | *:p | *:r" mode="abbreviation rules">
        <xsl:apply-templates mode="#current"/>
    </xsl:template>
    
    <xsl:template match="*:tr" mode="abbreviation">
        <xsl:variable name="abbreviation">
            <xsl:apply-templates select="*:tc[1]" mode="#current"/>
        </xsl:variable>
        <xsl:variable name="description">
            <xsl:apply-templates select="*:tc[2]" mode="#current"/>
        </xsl:variable>
        <AA>
            <short><xsl:value-of select="normalize-space($abbreviation)"/></short>
            <long><xsl:value-of select="normalize-space($description)"/></long>
        </AA>
    </xsl:template>
    
    <xsl:template match="*:tr[*:tc[1][matches(.,'H&lt;n&gt;')]]" mode="abbreviation" priority="2">
        <xsl:for-each select="1 to 15">
            <AA>
                <short>
                    <xsl:value-of select="concat('H',current())"/>
                </short>
                <long>
                    <xsl:value-of select="concat('Heading ',current())"/>
                </long>
            </AA>    
        </xsl:for-each>
    </xsl:template>
    
    
    <xsl:template match="*:tr[*:tc[1][matches(.,'Rule ID')]]" mode="rules" priority="2">
        <!-- ignore header row -->
        <xsl:if test="parent::node()/preceding-sibling::*:p[1][matches(.,'(General|Schema) Design rules','i')]">
            <xsl:text>
</xsl:text>
            <xsl:comment><xsl:value-of select="parent::node()/preceding-sibling::*:p[1]"/></xsl:comment>
        </xsl:if>
    </xsl:template>
    
    <xsl:template match="*:tr[*:tc[1][not(matches(.,'\W[SG]D-\d\d'))]]" mode="rules"/>
    
    <xsl:template match="*:tr[*:tc[1][matches(.,'\W[SG]D-\d\d')]]" mode="rules">
        
        <st96rule>
            <xsl:attribute name="id">
                <xsl:variable name="rule-id">
                    <xsl:apply-templates select="*:tc[1]" mode="#current"/>
                </xsl:variable>
                <xsl:value-of select="replace($rule-id,'^.*\[([^\]]+)\].*$','$1')"/>
            </xsl:attribute>
            <xsl:if test="*:tc[2][matches(.,'^\s*Deleted\s*$')]">
                <xsl:attribute name="deleted">true</xsl:attribute>
            </xsl:if>
            <xsl:apply-templates select="*:tc[2][not(matches(.,'^\s*Deleted\s*$'))]" mode="#current"/>
        </st96rule>
    </xsl:template>
    
    <xsl:template match="*:t" mode="abbreviation rules">
        <xsl:value-of select="."/>
    </xsl:template>
    
    <xsl:template match="*:del" priority="1000" mode="#all" />
    <xsl:template match="*:ins | *:hyperlink | *:moveTo" priority="100"  mode="#all">
        <xsl:apply-templates mode="#current"/>
    </xsl:template>
    
    <xsl:template match="*:instrText|*:delInstrText"  mode="#all">
        <!-- field code and deleted field code -->
    </xsl:template>
    
    <xsl:template match="*:tr[*:trPr/*:del] | *:delText | *:moveFrom"  mode="#all"/>
    <!-- deleted rows; deltext should be in del, but just in case..  -->

    <xsl:template match="*:commentRangeStart|*:commentRangeEnd"  mode="#all"/>
    <xsl:template match="*:commentReference"  mode="#all"/>
    
    <xsl:template match="*:p/*:customXml |
        *:p/*:smartTag|
        *:p/*:sdt|
        *:p/*:proofErr|
        *:p/*:permStart|
        *:p/*:permEnd |
*:p/*:bookmarkStart |
*:p/*:bookmarkEnd |
*:p/*:moveFromRangeStart |
*:p/*:moveFromRangeEnd |
*:p/*:moveToRangeStart |
*:p/*:moveToRangeEnd |
*:p/*:commentRangeStart |
*:p/*:commentRangeEnd |
*:p/*:customXmlInsRangeStart |
*:p/*:customXmlInsRangeEnd |
*:p/*:customXmlDelRangeStart |
*:p/*:customXmlDelRangeEnd |
*:p/*:customXmlMoveFromRangeStart |
*:p/*:customXmlMoveFromRangeEnd |
*:p/*:customXmlMoveToRangeStart |
*:p/*:customXmlMoveToRangeEnd |
*:p/*:subDoc |
*:p/*:fldSimple"  mode="#all">
    </xsl:template>
    
</xsl:stylesheet>