<?xml version="1.0" encoding="UTF-8"?>
<!--
    ST.96 conformance checking using Schematron
    Developed by: USPTO, Enterprise Data Architecture Division (EDAD)
	Point of Contact: Narith Tith, narith.tith@uspto.gov, 571-272-5458
-->
<xsl:stylesheet version="2.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:svrl="http://purl.oclc.org/dsdl/svrl"
                xmlns:sch="http://purl.oclc.org/dsdl/schematron"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:fnx="http://www.example.com/fnx">
  <xsl:import href="common_style.xsl"/>

  <!-- Either svrlSingleFile or svrlDir must be set -->
  <xsl:param name="svrlSingleFile"/>
  <xsl:param name="svrlDir"/>

  <!-- When svrlSingleFile is set, these params must be set. -->
  <xsl:param name="xsdSingleFile"/>

  <!-- When svrlDir is set, these params must be set. -->
  <xsl:param name="inputDirShort"/>

  <xsl:variable name="st96rulesFile" select="concat($devDirForwardSlashes, '/st96rules.xml')"/>
  <xsl:variable name="schFile" select="concat($devDirForwardSlashes, '/schemas/wipo_xsd.sch')"/>
  <xsl:variable name="svrlSingleFileFullPath" select="concat($devDirForwardSlashes, '/', $svrlSingleFile)"/>
  <xsl:variable name="svrlDirFullPath" select="concat($devDirForwardSlashes, '/', $svrlDir)"/>

  <!-- See also parallel Approved Exceptions settings and functions in
       wipo_xsd.sch.  (Alas, importing common file to Schematron seems
       to be impossible.)

       NOTE: be sure to set kWhereToApproveExceptions in both places
       consistently.
  -->
  <xsl:variable name="kWhereToApproveExceptions" select="'InReport'"/>
  <!--
      Configure kWhereToApproveExceptions to one of the following values:
      'InReport'
          Allow summary.html to report which XSDs violations are approved.
      'InSchematronRules'
          Trim approved exception violation notices immediately in Schematron.
  -->
  <xsl:variable name="approvedExceptions"
                select="if (doc-available('../ApprovedExceptions.xml')) then doc('../ApprovedExceptions.xml') else false()"/>

  <xsl:function name="fnx:is-exception" as="xs:boolean">
    <xsl:param name="ruleID" as="xs:string"/>
    <xsl:param name="url" as="xs:string"/>
    <xsl:variable name="subdirSlashXSD" as="xs:string" select="fnx:ShortenName($url)"/>
    <xsl:variable name="subdirName" as="xs:string" select="substring-before($subdirSlashXSD, '/')"/>
    <xsl:variable name="xsdFileName" as="xs:string" select="substring-after($subdirSlashXSD, '/')"/>


    <xsl:variable name="result">
      <xsl:value-of select="($kWhereToApproveExceptions = 'InReport')
                            and $approvedExceptions
                            and boolean($approvedExceptions/ApprovedExceptions/ApprovedException[@ruleID=$ruleID]/xsd[@subdir=$subdirName][@file=$xsdFileName])"/>
    </xsl:variable>
<!--
    <xsl:message>
      ***fnx:is-exception()***&#xa;
      ruleID=<xsl:value-of select="$ruleID"/>;&#xa;
      url=<xsl:value-of select="$url"/>;&#xa;
      subdirName=<xsl:value-of select="$subdirName"/>;&#xa;
      xsdFileName=<xsl:value-of select="$xsdFileName"/>
      result=<xsl:value-of select="$result"/>
    </xsl:message>
-->
    <xsl:value-of select="$result"/>
  </xsl:function>



  <xsl:output encoding="UTF-8" method="html" indent="yes"/>
  <xsl:strip-space elements="*"/>

  <xsl:include href="common.xsl"/>


  <xsl:template match="/">
    <html>
      <head>
        <title>ST96XSDValidator Validation Results - ST.96 Conformance Checking via Schematron</title>
         <xsl:sequence select="fnx:include-css()"/>
        <script type="text/javascript">
          <xsl:text disable-output-escaping="yes">
          function hideShowRemaining(st96RuleId, startNum, endNum) {
            var which = 'none';
            var hideShowEl = document.getElementById(st96RuleId + '_more');
            var hideShowText = hideShowEl.innerHTML;
            if (hideShowText == 'Show remaining violations') {
              which = '';
              hideShowEl.innerHTML = 'Hide remaining violations';
            } else {
              which = 'none';
              hideShowEl.innerHTML = 'Show remaining violations';
            }
            for (var i = startNum; i &lt; endNum; i++) {
              document.getElementById(st96RuleId + '_' + i).style.display = which;
            }
          }
          </xsl:text>
        </script>
      </head>
      <body>
		  <h1 align="center" class="noWrap"><xsl:call-template name="genXSDValidatorVersionInfo"/> Validation Results </h1>
		  <h2 align="center" class="noWrap">ST.96 Conformance Checking via Schematron</h2>     
		   <xsl:sequence select="fnx:time-run()"/>
         <div class="drcVersionDiv">
		  <hr/>
		  <xsl:call-template name="genDrcVersionInfo"/>
          There are <xsl:value-of
          select="$totalST96ruleCount"/><xsl:text> </xsl:text><a
          href="../../doc/03-96-i.pdf" target="_blank">ST96 Design Rules And Conventions</a> (DRC) rules:
          <xsl:value-of select="$gdST96ruleCount"/> General Design
          (GD) rules and <xsl:value-of select="$sdST96ruleCount"/>
          Schema Design (SD) rules.
          <p>See also:  [<a href="integritycheck.html" target="_blank">Schema Integrity Check Report</a>] and [<a href="namelist.html" target="_blank">Component Name Report</a>]</p>
          <hr/>
        </div>

        <h2>
          <xsl:text>Conformance Checking of </xsl:text>
          <xsl:value-of select="$numXSDs"/>
          <xsl:if test="$xsdSingleFile"> XSD: </xsl:if>
          <xsl:if test="not($xsdSingleFile)"> XSDs: </xsl:if>
          <span class="code">
            <xsl:value-of select="if ($xsdSingleFile) then $xsdSingleFile else $inputDirShort"/>
          </span>
        </h2>

        <div class="summaryDiv">
          <div>
            <span class="rulesSummaryTally"><xsl:value-of select="$autoCount"/> rules</span>
            <xsl:text> are checked automatically: </xsl:text>
            <span class="rulesSummaryTally"><xsl:value-of select="$auto-passCount"/> rules </span>
            <span class="passSpan">passed (<span class="ruleStatusIcon">&#10003;</span>)</span>
            <xsl:text>; </xsl:text>
            <span class="rulesSummaryTally"><xsl:value-of select="$auto-warnCount"/> rules </span>
            <span class="warnSpan">warned (<span class="ruleStatusIcon">W</span>)</span>
            <xsl:text>; </xsl:text>
            <xsl:value-of select="$auto-failCount"/>
            <xsl:text> rules </xsl:text>
            <span class="failSpan">failed <span class="ruleStatusIcon">(X)</span></span>
          </div>
          <div>
            <span class="rulesSummaryTally"><xsl:value-of select="$partialCount"/> rules</span>
            <xsl:text> are partially automated: </xsl:text>
            <span class="rulesSummaryTally"><xsl:value-of select="$partial-passCount"/> rules </span>
            <span class="manualSpan"> partially passed and require manual check (<span class="ruleStatusIcon">M*</span>)</span>
            <xsl:text>; </xsl:text>
            <span class="rulesSummaryTally"><xsl:value-of select="$partial-warnCount"/> rules </span>
            <span class="warnSpan">warned (<span class="ruleStatusIcon">W*</span>)</span>
            <xsl:text>; </xsl:text>
            <xsl:value-of select="$partial-failCount"/>
            <xsl:text> rules </xsl:text>
            <span class="failSpan">failed <span class="ruleStatusIcon">(X*)</span></span>
          </div>
          <div>
            <span class="rulesSummaryTally"><xsl:value-of select="$manualCount"/> rules</span> must be checked <span class="manualSpan">manually <span class="ruleStatusIcon">(M)</span></span>
          </div>
          <div class="tallyDiv">
            <span class="rulesSummaryTally tallySpan"><xsl:value-of select="$unaddressedCount"/> rules</span> have not yet been addressed <span class="ruleStatusIcon">(?)</span>;
            <span class="rulesSummaryTally"><xsl:value-of select="$deletedCount"/></span> <xsl:value-of select="if ($deletedCount = 1) then ' rule has' else ' rules have'"/> been <span class="deletedSpan">deleted <span class="ruleStatusIcon">(D)</span></span>
          </div>
          <div style="margin-bottom: 16px">
            <span class="rulesSummaryTally"><xsl:value-of select="$totalST96ruleCount"/> rules</span> total
          </div>
        </div>

        <!-- For Debugging/Diagnostics Only
        <h2>resultsPass1</h2>
        <xsl:call-template name="DebugInfo">
          <xsl:with-param name="xmlInfo" select="$resultsPass1/patterns/pattern[position() &lt; 120]"/>
        </xsl:call-template>
        <h2>resultsPass2</h2>
        <xsl:call-template name="DebugInfo">
          <xsl:with-param name="xmlInfo" select="$resultsPass2/patterns/pattern[position() &lt; 120]"/>
        </xsl:call-template>
        -->

        <table>
          <colgroup>
            <col width="72px"/>
            <col width="32px"/>
          </colgroup>
          <tr>
            <th class="ruleIdTh">Rule Id</th>
            <th title="Status: Pass (&#10003;); Fail (X); Manual (M); Unaddressed (?)" class="statusTh"></th>
            <th>Description / Results</th>
          </tr>
          <xsl:for-each select="$st96rules/st96rule">
            <xsl:variable name="st96ruleId" select="@id"/>
            <xsl:variable name="resultPattern" select="$resultsPass2/patterns/pattern[@st96ruleId eq $st96ruleId]"/>
            <xsl:variable name="method" select="fnx:calc_pattern_method($st96ruleId)"/>

            <xsl:variable name="approvedViolationResultElementsOfThisRule"
                          select="for $r in $resultPattern/result
                                  return if (fnx:is-exception($st96ruleId, $r/@uri))
                                  then $r else ()"/>

            <xsl:variable name="violationResultElementsOfThisRule"
                          select="for $r in $resultPattern/result
                                  return if (not(fnx:is-exception($st96ruleId, $r/@uri)))
                                  then $r else ()"/>

            <xsl:variable name="xsdsViolatingThisRule"
                          select="for $uri in distinct-values($resultPattern/result/@uri)
                                  return if (not(fnx:is-exception($st96ruleId, $uri)))
                                  then fnx:ShortenName($uri) else ()"/>
            <xsl:variable name="numApprovedViolationsOfThisRule" select="count($approvedViolationResultElementsOfThisRule)"/>
            <xsl:variable name="numViolationsOfThisRule" select="count($violationResultElementsOfThisRule)"/>
            <xsl:variable name="numXSDsViolatingThisRule" select="count($xsdsViolatingThisRule)"/>

            <xsl:variable name ="calculatedStatus">
              <xsl:choose>
                <xsl:when test="$numViolationsOfThisRule > 0">
                  <xsl:value-of select="$resultPattern/@status"/>
                </xsl:when>
                <xsl:otherwise>success</xsl:otherwise>
              </xsl:choose>
            </xsl:variable>

            <xsl:variable name="isDeleted" select="$st96rules/st96rule[@id=$st96ruleId and @deleted='true']"/>
            <xsl:variable name="trClass" select="if ($isDeleted) then 'deletedTr' else fnx:make_tr_class($calculatedStatus, $method)"/>
            <tr class="{$trClass}">
              <xsl:variable name="deletedClass" select="if ($isDeleted) then 'deletedRule' else ''"/>
              <td class="ruleIdTd {$deletedClass}">
                <xsl:value-of select="$st96ruleId"/>
              </td>
              <td title="Status: Pass (&#10003;); Warn (W); Fail (F); Manual (M); Deleted (D); Unaddressed (?)" class="methodSummaryTd">
                <xsl:choose>
                 <xsl:when test="$method eq 'Partial'">
                    <xsl:choose>
                      <xsl:when test="$calculatedStatus eq 'failure'"><span class="failureIcon">X*</span></xsl:when>
                      <xsl:when test="$calculatedStatus eq 'warning'"><span class="warningIcon">W*</span></xsl:when>
                      <xsl:otherwise><span class="manualIcon">M*</span></xsl:otherwise>
                    </xsl:choose>
                  </xsl:when>
                  <xsl:when test="$method eq 'Manual'"><span class="manualIcon">M</span></xsl:when>
                  <xsl:when test="$isDeleted"><span class="deletedIcon">D</span></xsl:when>
                  <xsl:when test="$method eq 'Auto'">
                    <xsl:choose>
                      <xsl:when test="$calculatedStatus eq 'failure'"><span class="failureIcon">X</span></xsl:when>
                      <xsl:when test="$calculatedStatus eq 'warning'"><span class="warningIcon">W</span></xsl:when>
                      <xsl:otherwise><span class="passIcon">&#10003;</span></xsl:otherwise>
                    </xsl:choose>
                  </xsl:when>
                  <xsl:when test="$method eq 'Unaddressed'"><span class="unaddressedIcon">?</span></xsl:when>
                  <xsl:otherwise>[???]</xsl:otherwise>
                </xsl:choose>
              </td>
              <td colspan="2" class="descriptionTd">
                <xsl:choose>
                  <xsl:when test="$isDeleted">
                    <i>Deleted</i>
                  </xsl:when>
                  <xsl:otherwise><xsl:copy-of select="."/></xsl:otherwise>
                </xsl:choose>
              </td>
            </tr>

            <xsl:if test="count($resultPattern) > 1">
              <tr>
                <td colspan="8" style="color: red; font-size: 24pt; font-weight: bold">
                  <xsl:value-of select="$st96ruleId"/>: More than one $resultPattern!
                </td>
              </tr>
            </xsl:if>

            <xsl:if test="$calculatedStatus eq 'failure' or $calculatedStatus eq 'warning'">
              <xsl:if test="$calculatedStatus != 'success'">
                <xsl:choose>
                  <xsl:when test="$numXSDs &gt; 10 and $numXSDsViolatingThisRule = $numXSDs">
                    <tr class="{$trClass}">
                      <td/>
                      <td/>
                      <td colspan="2" class="violationHeaderTd">All <xsl:value-of select="$numXSDs"/> XSDs violate rule <xsl:value-of select="$st96ruleId"/>.</td>
                    </tr>
                  </xsl:when>
                  <xsl:when test="$numViolationsOfThisRule &gt; 10">
                    <tr class="{$trClass}">
                      <td/>
                      <td/>
                      <td class="violationHeaderTd"><xsl:value-of select="$numXSDsViolatingThisRule"/> XSDs violate rule <xsl:value-of select="$st96ruleId"/>.  Here are the first 10 violations:</td>
                      <td class="violationHeaderTd">Violation details:</td>
                    </tr>
                    <xsl:for-each select="1 to 10">
                      <xsl:variable name="i" select="."/>
                      <xsl:variable name="r" select="$violationResultElementsOfThisRule[$i]"/>
                      <xsl:variable name="uri" select="$r/@uri"/>
                      <xsl:variable name="shortenedUri" select="fnx:ShortenName($uri)"/>
                      <tr class="{$trClass}">
                        <td/>
                        <td/>
                        <td class="fileNameTd"><a href="{$inputDirShort}\{$shortenedUri}" target="_blank"><xsl:value-of select="$shortenedUri"/></a></td>
                        <td class="reportTextTd"><xsl:value-of select="$r/text()"/></td>
                      </tr>
                    </xsl:for-each>
                    <tr class="{$trClass}">
                      <td/>
                      <td/>
                      <td class="showRemainingTd">[ <a id="{$st96ruleId}_more" class="showRemainingA" href="javascript:hideShowRemaining('{$st96ruleId}', 10, {10+count($resultPattern/result[10]/following-sibling::result)})">Show remaining violations</a> ]</td>
                      <td/>
                    </tr>
                    <xsl:for-each select="10 to $numViolationsOfThisRule">
                      <xsl:variable name="i" select="."/>
                      <xsl:variable name="r" select="$violationResultElementsOfThisRule[$i]"/>

                      <!--<xsl:variable name="resultNum" select="count(preceding-sibling::result)"/>-->
                      <xsl:variable name="resultNum" select="$i"/>
                      <xsl:variable name="uri" select="$r/@uri"/>
                      <xsl:variable name="shortenedUri" select="fnx:ShortenName($uri)"/>
                      <tr id="{$st96ruleId}_{$resultNum}" class="{$trClass}" style="display: none">
                        <td/>
                        <td/>
                        <td class="fileNameTd"><a href="{$inputDirShort}\{$shortenedUri}" target="_blank"><xsl:value-of select="$shortenedUri"/></a></td>
                        <td class="reportTextTd"><xsl:value-of select="$r/text()"/></td>
                      </tr>
                    </xsl:for-each>
                  </xsl:when>
                  <xsl:otherwise>
                    <tr class="{$trClass}">
                      <td/>
                      <td/>
                      <td class="violationHeaderTd">These <xsl:value-of select="if ($numXSDsViolatingThisRule &gt; 3) then $numXSDsViolatingThisRule else ''"/> XSDs violate rule <xsl:value-of select="$st96ruleId"/>:</td>
                      <td class="violationHeaderTd">Violation details:</td>
                    </tr>
                    <xsl:for-each select="1 to $numViolationsOfThisRule">
                      <xsl:variable name="i" select="."/>
                      <xsl:variable name="r" select="$violationResultElementsOfThisRule[$i]"/>
                      <xsl:variable name="uri" select="$r/@uri"/>
                      <xsl:variable name="shortenedUri" select="fnx:ShortenName($uri)"/>
                      <tr class="{$trClass}">
                        <td/>
                        <td/>
                        <td class="fileNameTd"><a href="{$inputDirShort}\{$shortenedUri}" target="_blank"><xsl:value-of select="$shortenedUri"/></a></td>
                        <td class="reportTextTd"><xsl:value-of select="$r/text()"/></td>
                      </tr>
                    </xsl:for-each>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:if>
            </xsl:if>
            <!-- List Approved Exceptions -->
            <xsl:if test="$numApprovedViolationsOfThisRule > 0">
              <tr class="approvedExceptionTr">
                <td class="{$trClass}"/>
                <td class="{$trClass}"/>
                <td class="violationHeaderTd">These <xsl:value-of select="if ($numApprovedViolationsOfThisRule &gt; 3) then $numApprovedViolationsOfThisRule else ''"/> XSDs violate rule <xsl:value-of select="$st96ruleId"/> but are approved exceptions:</td>
                <td class="violationHeaderTd">Violation details:</td>
              </tr>
              <xsl:for-each select="$approvedViolationResultElementsOfThisRule">
                <xsl:variable name="uri" select="@uri"/>
                <xsl:variable name="shortenedUri" select="fnx:ShortenName($uri)"/>
                <tr class="approvedExceptionTr">
                  <td class="{$trClass}"/>
                  <td class="{$trClass}"/>
                  <td class="fileNameTd"><a href="{$inputDirShort}\{$shortenedUri}" target="_blank"><xsl:value-of select="$shortenedUri"/></a></td>
                  <td class="reportTextTd"><xsl:value-of select="text()"/></td>
                </tr>
              </xsl:for-each>
            </xsl:if>
          </xsl:for-each>
        </table>
      </body>
    </html>
  </xsl:template>

  <xsl:function name="fnx:ShortenName">
   <xsl:param name="uri"/>
    <xsl:value-of select="replace(substring-after($uri, 'dev/svrl/'), '.svrl', '.xsd')"/>
  </xsl:function>

  <xsl:function name="fnx:except-any-atomic" as="xs:anyAtomicType*">
    <xsl:param name="a1" as="xs:anyAtomicType*"/>
    <xsl:param name="a2" as="xs:anyAtomicType*"/>
    <xsl:sequence select="distinct-values($a1[not(.=$a2)])"/>
  </xsl:function>

</xsl:stylesheet>
