<?xml version="1.0" encoding="UTF-8"?>
<!--
    ST.96 conformance checking using Schematron
    Check schema integrity
     - unused imports/includes
     - unresolveable imports/includes
     - unused namespaces
     - orphan components
     - duplication of common components

    Developed by: USPTO, Enterprise Data Architecture Division (EDAD)
	Point of Contact: Narith Tith, narith.tith@uspto.gov, 571-272-5458
-->
<xsl:stylesheet version="2.0" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:dty="http://www.datypic.com" exclude-result-prefixes="xsd xsl dty">
   <xsl:output encoding="UTF-8" method="xml" indent="yes"/>
   <xsl:strip-space elements="*"/>
   <xsl:param name="reportInDir"/>
   <xsl:template match="xsd:schema">
      <xsl:variable name="integrity-info" select="doc('../integritycheck/refs.configxml')/names"/>
      <xsl:variable name="nsUsedPrefixesElements" select="//*/substring-before(name(),':')" as="xsd:string*"/>
      <xsl:variable name="nsUsedPrefixesAttrs" select="//(@type|@ref|@itemType|@refer|@base)/substring-before(., ':')" as="xsd:string*"/>
      <xsl:variable name="nsUsedPrefixesMemberTypes" select="for $val in //@memberTypes/tokenize(.,'\s+') return substring-before($val, ':')" as="xsd:string*"/>
      <xsl:variable name="nsUsedPrefixes" select="distinct-values(($nsUsedPrefixesElements, $nsUsedPrefixesAttrs, $nsUsedPrefixesMemberTypes))"/>
      <xsl:variable name="tns" select="xsd:schema/@targetNamespace"/>
      <xsl:variable name="nsUnusedPrefixes" as="xsd:string*">
         <xsl:for-each select="xsd:schema/namespace::*">
            <xsl:variable name="nsPrefix" select="name()"/>
            <xsl:variable name="nsURI" select="."/>
            <xsl:if test="$nsPrefix != 'xml' and $nsURI != $tns and empty(index-of($nsUsedPrefixes, $nsPrefix))">
               <xsl:value-of select="$nsPrefix"/>
            </xsl:if>
         </xsl:for-each>
      </xsl:variable>
      <xsl:variable name="UsedNamesAttrs" select="//(@type|@ref|@itemType|@refer|@base)/dty:resolve-name-in-attr(.)" as="xsd:QName*"/>
      <xsl:variable name="UsedNamesMemberTypes" as="xsd:QName*">
         <xsl:for-each select="//@memberTypes">
            <xsl:variable name="el" select=".."/>
            <xsl:for-each select="tokenize(.,'\s+')">
               <xsl:sequence select="resolve-QName(.,$el)"/>
            </xsl:for-each>
         </xsl:for-each>
      </xsl:variable>
      <xsl:variable name="UsedNames" select="distinct-values(($UsedNamesAttrs, $UsedNamesMemberTypes))" as="xsd:QName*"/>
      <xsl:variable name="names-one-level-down" select="dty:names-one-level-down(.)" as="xsd:QName*"/>
      <xsl:variable name="NamesNotCoveredDirectly" select="$UsedNames[not(namespace-uri-from-QName(.) = 'http://www.w3.org/2001/XMLSchema')][not(. = $names-one-level-down)]" as="xsd:QName*"/>
      <integrity-info file="{base-uri()}" namespace="{@targetNamespace}">
         <xsl:if test="count($nsUnusedPrefixes) > 0">
            <unused-prefixes prefixes="{$nsUnusedPrefixes}"/>
         </xsl:if>
         <!-- get the names used one level deep.  if they are all accounted for, any imported file that doesn't have any names 
           is unnecessary.  This eliminates the performance issue associated with unnecessary imports of huge 
           schemas like PhraseType -->
         <xsl:if test="exists($NamesNotCoveredDirectly)">
            <missing-import names="{string-join(for $n in $NamesNotCoveredDirectly return string($n),', ')}"/>
         </xsl:if>
         <xsl:apply-templates select="xsd:import|xsd:include">
            <xsl:with-param name="UsedNames" select="$UsedNames"/>
            <xsl:with-param name="NamesNotCoveredDirectly" select="$NamesNotCoveredDirectly"/>
         </xsl:apply-templates>
         <xsl:if test="not(contains(base-uri(),'/Document/'))">
            <xsl:apply-templates select="xsd:element|xsd:attribute|xsd:complexType|xsd:simpleType" mode="orphan">
               <xsl:with-param name="tns" select="@targetNamespace"/>
               <xsl:with-param name="integrity-info" select="$integrity-info"/>
            </xsl:apply-templates>
         </xsl:if>
         <xsl:if test="not(@targetNamespace = ('http://www.wipo.int/standards/XMLSchema/ST96/Common','urn:us:gov:doc:uspto:common'))">
            <xsl:apply-templates select="xsd:element|xsd:attribute|xsd:complexType|xsd:simpleType" mode="dup-common">
               <xsl:with-param name="tns" select="@targetNamespace"/>
               <xsl:with-param name="integrity-info" select="$integrity-info"/>
            </xsl:apply-templates>
         </xsl:if>
      </integrity-info>
   </xsl:template>
   <xsl:template match="xsd:import|xsd:include">
      <xsl:param name="UsedNames"/>
      <xsl:param name="NamesNotCoveredDirectly"/>
      <xsl:variable name="ImportedFile" select="resolve-uri(@schemaLocation,document-uri(/))"/>
      <!--<xsl:message select="$ImportedFile"/>-->
      <xsl:choose>
         <xsl:when test="doc-available($ImportedFile)">
            <xsl:variable name="imported" select="doc($ImportedFile)"/>
            <xsl:variable name="DefinedNames" select="dty:names-in-schema($imported)"/>
            <xsl:if test="not($DefinedNames = $UsedNames)">
               <xsl:choose>
                  <xsl:when test="empty($NamesNotCoveredDirectly)">
                     <!-- no need to go recursively checking in this import because all names are accounted for
                          in other imports - saves a lot of time -->
                     <import-unused element="{local-name(.)}" imported-file="{@schemaLocation}"/>
                  </xsl:when>
                  <xsl:otherwise>
                     <xsl:variable name="used-recursive" select="dty:name-used-recursive($imported,$ImportedFile,$UsedNames)"/>
                     <xsl:choose>
                        <xsl:when test="exists($used-recursive)">
                           <import-unused-directly element="{local-name(.)}" imported-file="{@schemaLocation}"/>
                        </xsl:when>
                        <xsl:otherwise>
                           <import-unused element="{local-name(.)}" imported-file="{@schemaLocation}"/>
                        </xsl:otherwise>
                     </xsl:choose>
                  </xsl:otherwise>
               </xsl:choose>
            </xsl:if>
         </xsl:when>
         <xsl:otherwise>
            <import-file-not-found element="{local-name(.)}" imported-file="{@schemaLocation}"/>
         </xsl:otherwise>
      </xsl:choose>
   </xsl:template>
   <xsl:template match="text()"/>
   <xsl:template match="xsd:element|xsd:attribute|xsd:complexType|xsd:simpleType" mode="orphan">
      <xsl:param name="integrity-info"/>
      <xsl:param name="tns"/>
      <xsl:if test="not($integrity-info/ref[@name = current()/@name and @ns=$tns])">
         <orphan type="{local-name(.)}" name="{@name}"/>
      </xsl:if>
   </xsl:template>
   <xsl:template match="xsd:element|xsd:attribute|xsd:complexType|xsd:simpleType" mode="dup-common">
      <xsl:param name="integrity-info"/>
      <xsl:if test="$integrity-info/common-name[@name = current()/@name]">
         <dup-common type="{local-name(.)}" name="{@name}" namespace="{$integrity-info/common-name[@name = current()/@name]/@ns}"/>
      </xsl:if>
   </xsl:template>
   <xsl:function name="dty:resolve-name-in-attr" as="xsd:QName">
      <xsl:param name="prefixed-name" as="attribute()"/>
      <!-- for @ref, @type, @base, etc. -->
      <xsl:sequence select="resolve-QName($prefixed-name,$prefixed-name/parent::*)"/>
   </xsl:function>
   <xsl:function name="dty:schema-name" as="xsd:QName">
      <xsl:param name="unprefixed-name" as="attribute()"/>
      <!-- for @name -->
      <xsl:sequence select="QName($unprefixed-name/ancestor::xsd:schema/@targetNamespace,$unprefixed-name)"/>
   </xsl:function>
   <xsl:function name="dty:names-in-schema" as="xsd:QName*">
      <xsl:param name="doc"/>
      <xsl:sequence select="$doc//@name/dty:schema-name(.)"/>
   </xsl:function>
   <xsl:function name="dty:name-used-recursive" as="xsd:boolean*">
      <xsl:param name="doc"/>
      <xsl:param name="used-uris"/>
      <xsl:param name="used-names"/>
      <xsl:choose>
         <xsl:when test="not($doc/xsd:schema/(xsd:import|xsd:include))">
            <xsl:sequence select="()"/>
         </xsl:when>
         <xsl:when test="$used-names = dty:names-one-level-down($doc)">
            <xsl:sequence select="true()"/>
         </xsl:when>
         <xsl:otherwise>
            <!-- go down one more level -->
            <xsl:for-each select="$doc/xsd:schema/(xsd:import|xsd:include)">
               <xsl:variable name="ImportedFile" select="resolve-uri(@schemaLocation,document-uri($doc))"/>
               <xsl:if test="not($ImportedFile = $used-uris) and doc-available($ImportedFile)">
                  <xsl:variable name="imported" select="doc($ImportedFile)"/>
                  <xsl:sequence select="dty:name-used-recursive($imported, ($used-uris,$ImportedFile),$used-names)"/>
               </xsl:if>
            </xsl:for-each>
         </xsl:otherwise>
      </xsl:choose>
   </xsl:function>
   <xsl:function name="dty:names-one-level-down" as="xsd:QName*">
      <xsl:param name="doc"/>
      <xsl:for-each select="$doc//(xsd:import|xsd:include)">
         <xsl:variable name="ImportedFile" select="resolve-uri(@schemaLocation,document-uri(root($doc)))"/>
         <xsl:if test="doc-available($ImportedFile)">
            <xsl:variable name="imported" select="doc($ImportedFile)"/>
            <xsl:sequence select="dty:names-in-schema($imported)"/>
         </xsl:if>
      </xsl:for-each>
   </xsl:function>
</xsl:stylesheet>
