<?xml version="1.0" encoding="UTF-8"?>
<!--
	ST.96 conformance checking using Schematron
    Developed by: USPTO, Enterprise Data Architecture Division (EDAD)
	Point of Contact: Narith Tith, narith.tith@uspto.gov, 571-272-5458
-->
<schema queryBinding="xslt2" schemaVersion="1.0" xmlns="http://purl.oclc.org/dsdl/schematron" xmlns:sch="http://purl.oclc.org/dsdl/schematron" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:fnx="http://www.example.com/fnx" xmlns:d="http://www.example.com/data" xmlns:mathml="http://www.w3.org/1998/Math/MathML" xmlns:com="http://www.wipo.int/standards/XMLSchema/ST96/Common" xmlns:tbl="http://www.oasis-open.org/tables/exchange/1.0" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
  <title>WIPO ST.96 Conformance Validation</title>
 <ns prefix="fnx" uri="http://www.example.com/fnx"/>
   <ns prefix="d" uri="http://www.example.com/data"/>
   <ns prefix="m" uri="http://www.w3.org/1998/Math/MathML"/>
   <ns prefix="tbl" uri="http://www.oasis-open.org/tables/exchange/1.0"/>
   <ns prefix="com" uri="http://www.wipo.int/standards/XMLSchema/ST96/Common"/>
   <ns prefix="xsd" uri="http://www.w3.org/2001/XMLSchema"/>
   <let name="localFile" value="replace(base-uri(.),'^.*[\\/](.*)','$1')"/>
   <let name="inSubdirName" value="tokenize(base-uri(), '/')[last()-1]"/>
   <let name="inComponentSubdirName" value="tokenize(replace(base-uri(),'/Document/','/'), '/')[last()-1]"/>
   <let name="kRegexReleaseDir" value="'.*ST96/([^\\/]+)/.*'"/>
   <let name="kRegexReleaseVersion" value="'V([0-9]+)_([0-9]+)'"/>
   <let name="kRegexLocalFileVersion" value="'^[a-zA-Z0-9]+_V([0-9]+)_([0-9]+)\.[a-zA-Z0-9]+$'"/>
   <!-- See also parallel Approved Exceptions settings and functions in
       summary.sch.  (Alas, importing common file to Schematron seems
       to be impossible.)

       NOTE: be sure to set kWhereToApproveExceptions in both places
       consistently.
  -->
  <let name="kWhereToApproveExceptions" value="'InReport'"/>
  <!--
      Configure kWhereToApproveExceptions to one of the following values:
      'InReport'
          Allow summary.html to report which XSDs violations are approved.
      'InSchematronRules'
          Trim approved exception violation notices immediately in Schematron.
  -->
  <let name="kRepresentationTerms" value="('Amount', 'Category', 'Code', 'Date', 'Identifier',
              'Indicator', 'Measure', 'Name', 'Number', 'Percent',
              'Quantity', 'Rate', 'Text', 'Time', 'DateTime', 'URI')"/>
   <let name="kNamespaceWIPObase" value="'http://www.wipo.int/standards/XMLSchema/ST96/'"/>
   <let name="kCommonNamespace" value="concat($kNamespaceWIPObase, 'Common')"/>
   <let name="kPatentNamespace" value="concat($kNamespaceWIPObase, 'Patent')"/>
   <let name="kTrademarkNamespace" value="concat($kNamespaceWIPObase, 'Trademark')"/>
   <let name="kDesignNamespace" value="concat($kNamespaceWIPObase, 'Design')"/>
   <let name="kXMLSchemaNamespace" value="'http://www.w3.org/2001/XMLSchema'"/>
   <let name="kMathMLNamespace" value="'http://www.w3.org/1998/Math/MathML'"/>
   <let name="kOASISTableNamespace" value="'http://www.oasis-open.org/tables/exchange/1.0'"/>
   <let name="kIPOCommonNamespace" value="fnx:ipo-ns('common', 'ns')"/>
   <let name="kIPOPatentNamespace" value="fnx:ipo-ns('patent', 'ns')"/>
   <let name="kIPOTrademarkNamespace" value="fnx:ipo-ns('trademark', 'ns')"/>
   <let name="kIPODesignNamespace" value="fnx:ipo-ns('design', 'ns')"/>
   <let name="kIPOCommonNamespacePrefix" value="fnx:ipo-ns('common', 'ns-prefix')"/>
   <let name="prefixTable" value="fnx:buildPrefixTable(/xsd:schema)"/>
   <let name="approvedAcronyms" value="doc('../AcronymsAndAbbreviations.xml'), doc('../AcronymsAndAbbreviationsLocal.xml')"/>
   <let name="approvedExceptions" value="if (doc-available('../ApprovedExceptions.xml')) then doc('../ApprovedExceptions.xml') else false()"/>
   <let name="definitions" value="fnx:gatherDefinitionsRoot(/xsd:schema, 0)"/>
   <let name="connectingWords" value="doc('../ConnectingWords.xml')/Words/Word"/>
   <let name="legalReferencePrefixes" value="doc('../LegalReferencePrefixes.xml')/Prefixes/Prefix"/>
   <let name="namespaces" value="doc('../BaselineNamespaces.xml')/Namespaces/Namespace"/>
   <let name="lookupTable" value="fnx:buildLookupTable()"/>
   <let name="st3Codes" value="('AD','AE','AF','AG','AI','AL','AM','AN','AO','AP','AR','AT','AU','AW','AZ','BA','BB','BD','BE','BF','BG','BH','BI','BJ','BM','BN','BO','BR','BS','BT','BV','BW','BX','BY','BZ','CA','CD','CF','CG','CH','CI','CK','CL','CM','CN','CO','CR','CU','CV','CY','CZ','DE','DJ','DK','DM','DO','DZ','EA','EC','EE','EG','EH','EM','EP','ER','ES','ET','FI','FJ','FK','FO','FR','GA','GB','GC','GD','GE','GG','GH','GI','GL','GM','GN','GQ','GR','GS','GT','GW','GY','HK','HN','HR','HT','HU','IB','ID','IE','IL','IM','IN','IQ','IR','IS','IT','JE','JM','JO','JP','KE','KG','KH','KI','KM','KN','KP','KR','KW','KY','KZ','LA','LB','LC','LI','LK','LR','LS','LT','LU','LV','LY','MA','MC','MD','ME','MG','MK','ML','MM','MN','MO','MP','MR','MS','MT','MU','MV','MW','MX','MY','MZ','NA','NE','NG','NI','NL','NO','NP','NR','NZ','OA','OM','PA','PE','PG','PH','PK','PL','PT','PW','PY','QA','QZ','RO','RS','RU','RW','SA','SB','SC','SD','SE','SG','SH','SI','SK','SL','SM','SN','SO','SR','ST','SV','SY','SZ','TC','TD','TG','TH','TJ','TL','TM','TN','TO','TR','TT','TV','TW','TZ','UA','UG','US','UY','UZ','VA','VC','VE','VG','VN','VU','WO','WS','XN','YE','ZA','ZM','ZW')"/>

  <!--
  <let name="isFlattened"
       value="(count(/xsd:schema/xsd:element) + count(/xsd:schema/xsd:complexType) + count(/xsd:schema/xsd:simpleType) > 1)"/>
  -->

  <let name="isFlattened" value="($inSubdirName = 'FlattenedDocumentSchema')"/>
               
  <let name="kIPONonCommonNamespaces" value="($kIPOPatentNamespace,$kIPOTrademarkNamespace,$kIPODesignNamespace)"/>
  <let name="kNonCommonNamespaces" value="($kPatentNamespace,$kTrademarkNamespace,$kDesignNamespace)"/>

  <xsl:function name="fnx:is-exception" as="xs:boolean">
    <xsl:param name="ruleID" as="xs:string"/>

    <xsl:value-of select="($kWhereToApproveExceptions = 'InSchematronRules')
                          and $approvedExceptions
                          and boolean($approvedExceptions/ApprovedExceptions/ApprovedException[@ruleID=$ruleID]/xsd[not(@subdir) or @subdir=$inSubdirName][@file=$localFile])"/>
  </xsl:function>

  <xsl:function name="fnx:ipo-ns" as="xs:string?">
    <xsl:param name="area" as="xs:string"/>
    <xsl:param name="part" as="xs:string"/>
    
    <xsl:value-of select="$namespaces[@std='*'][@component = $area]/@*[name() = $part]"/>
  </xsl:function>

  <xsl:function name="fnx:xsdVersionFromFileName" as="xs:string?">
    <xsl:param name="fname" as="xs:string"/>
    
    <xsl:analyze-string select="$fname"
                        regex="^[a-zA-Z0-9]+_(V[0-9]+_[0-9]+)_D([0-9]+)\.[a-zA-Z0-9]+$">
      <xsl:matching-substring>
        <xsl:value-of select="regex-group(1)"/>
      </xsl:matching-substring>
    </xsl:analyze-string>
  </xsl:function>

  <xsl:function name="fnx:buildPrefixTable" as="element()">
    <!-- limitation: does not include prefix-namespace bindings from included/imported XSDs -->
    <xsl:param name="root" as="node()"/>
    <d:prefixBindings>
      <!-- http://stackoverflow.com/a/123005/290085 -->
      <xsl:for-each select="$root/namespace::*">
        <d:prefixBinding nsPrefix="{name()}" nsName="{.}"/>
      </xsl:for-each>
    </d:prefixBindings>
  </xsl:function>

  <xsl:function name="fnx:buildLookupTable" as="element()">
    <d:lookUp>
      <xsl:for-each select="distinct-values($definitions//d:namespace/@ns)">
        <xsl:variable name="ns" select="."/>
        <d:entry ns="{.}">
          <xsl:sequence select="$definitions//d:namespace[@ns = $ns]/*[local-name() != 'namespace']"/>
        </d:entry>
      </xsl:for-each>
    </d:lookUp>
  </xsl:function>

  <!-- Build first-pass structure of nested namespaces that will then
       be collapsed into a proper symbol table array -->
  <xsl:function name="fnx:gatherDefinitionsRoot" as="element()*">
    <xsl:param name="schemaRoot" as="node()"/>
    <xsl:param name="depth" as="xsd:integer"/>
    <d:defs>
      <xsl:sequence select="fnx:gatherDefinitions($schemaRoot, 0)"/>
    </d:defs>
  </xsl:function>

  <xsl:function name="fnx:gatherDefinitions" as="element()*">
    <xsl:param name="schemaRoot" as="node()?"/>
    <xsl:param name="depth" as="xsd:integer"/>

    <xsl:if test="$depth &lt; 4"> <!-- performance concession and loop avoidance -->
      <xsl:variable name="nsName" select="if ($schemaRoot/@targetNamespace) then $schemaRoot/@targetNamespace else 'DEFAULT'"/>

      <d:namespace ns="{$nsName}">
        <xsl:for-each select="$schemaRoot/(xsd:include|xsd:import)">
          <xsl:sequence select="fnx:gatherDefinitions(document(@schemaLocation)/xsd:schema, $depth + 1)"/>
        </xsl:for-each>

        <xsl:sequence select="$schemaRoot/(xsd:element|xsd:attribute|xsd:complexType|xsd:simpleType)"/>

      </d:namespace>
    </xsl:if>
  </xsl:function>

  <xsl:function name="fnx:same-position-LC" as="xs:boolean">
    <xsl:param name="str" as="xs:string"/>
    <xsl:param name="substr" as="xs:string"/>

    <xsl:variable name="strLC" select="lower-case($str)"/>
    <xsl:variable name="substrLC" select="lower-case($substr)"/>

    <xsl:value-of select="contains($str, $substr)
                          and lower-case(substring-after($str, $substr))
                            = substring-after($strLC, $substrLC)"/>
  </xsl:function>

  <!--
      Recursive function.  Do not call directly because does not
      special-case the start of string.  Call main entry point at
      fnx:find-improper-embedded-abbreviation-in-attribute() instead.
      It is assumed that the lower-case check of $abbrev against the
      beginning of the attribute name has already taken place prior to
      calling this function.
  -->
  <xsl:function name="fnx:find-improper-embedded-abbreviation-rest" as="xs:string*">
      <xsl:param name="nameRest" as="xs:string"/>
      <xsl:param name="abbrev" as="xs:string"/>
      <xsl:if test="$nameRest != ''">
         <xsl:variable name="nameRestLC" select="lower-case($nameRest)"/>
         <xsl:variable name="abbrevLC" select="lower-case($abbrev)"/>
         <xsl:if test="contains($nameRestLC, $abbrevLC)">
            <xsl:choose>
               <xsl:when test="not(fnx:same-position-LC($nameRest, $abbrev))">
                  <xsl:value-of select="$abbrev"/>
               </xsl:when>
               <xsl:otherwise>
                  <xsl:variable name="rest" select="substring-after($nameRest, $abbrev)"/>
                  <xsl:value-of select="fnx:find-improper-embedded-abbreviation-rest($rest, $abbrev)"/>
               </xsl:otherwise>
            </xsl:choose>
         </xsl:if>
      </xsl:if>
   </xsl:function>

  <xsl:function name="fnx:find-improper-embedded-abbreviation-in-attribute" as="xs:string*">
    <xsl:param name="attrName" as="xs:string"/>
    
    <xsl:variable name="attrNameLC" select="lower-case($attrName)"/>
    <xsl:for-each select="$approvedAcronyms/AAs/AA/short[string-length() > 2]">
      <xsl:variable name="abbrev" select="."/>
      <xsl:variable name="abbrevLC" select="lower-case($abbrev)"/>
      <xsl:choose>
        <!-- An abbreviation at the beginning of a name must be in lower-case. -->
        <xsl:when test="starts-with($attrNameLC, $abbrevLC) and not(starts-with($attrName, $abbrevLC))">
          <xsl:value-of select="$abbrev"/>
        </xsl:when>
        <!-- Check the rest of an attribute name that properly starts with a lower-cased abbreviation.   -->
        <xsl:when test="starts-with($attrNameLC, $abbrevLC) and starts-with($attrName, $abbrevLC)">
         <xsl:value-of select="fnx:find-improper-embedded-abbreviation-rest(substring-after($attrName, $abbrevLC), $abbrev)"/>
        </xsl:when>
        <!-- Check the entire attribute name without regard to the special start-of-name condition. -->
<xsl:otherwise>
               <xsl:value-of select="fnx:find-improper-embedded-abbreviation-rest($attrName, $abbrev)"/>
            </xsl:otherwise>
         </xsl:choose>
      </xsl:for-each>
   </xsl:function>
   <xsl:function name="fnx:find-improper-embedded-abbreviation-in-element-or-type" as="xs:string*">
      <xsl:param name="elName" as="xs:string"/>
      <xsl:for-each select="$approvedAcronyms/AAs/AA/short[string-length() > 2]">
         <xsl:variable name="abbrev" select="."/>
         <xsl:value-of select="fnx:find-improper-embedded-abbreviation-rest($elName, $abbrev)"/>
      </xsl:for-each>
   </xsl:function>
   <xsl:function name="fnx:words-from-name" as="xs:string*">
      <xsl:param name="name"/>
      <!-- eliminates acronyms; only checks words -->
      <xsl:sequence select="tokenize(replace(replace($name,'([A-Z]*)([A-Z][^A-Z]+)',' $2'),'[A-Z]+$',''),' ')"/>
   </xsl:function>
  <xsl:function name="fnx:getRepTerm" as="xs:string">
    <xsl:param name="name" as="xs:string"/>
    <xsl:choose>
      <!--
          Specially check DateTime and Time because their mutually
          identical ending foils the for-expression approache below.
      -->
      <xsl:when test="ends-with($name, 'DateTime')">DateTime</xsl:when>
      <xsl:when test="ends-with($name, 'Time')">Time</xsl:when>
      <xsl:otherwise>
      <xsl:value-of select="string-join(
                            (for $repTerm in $kRepresentationTerms
                             return
                               if (ends-with(lower-case($name),
                                             lower-case($repTerm))
                                   or
                                   ends-with(lower-case($name),
                                             lower-case(concat($repTerm, 'Type')))
                                  )
                               then $repTerm
                               else ''),
                               '')"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>

  <xsl:function name="fnx:allowedRepTerms" as="xsd:string*">
    <xsl:param name="n" as="node()"/>
    <!--
    <xsl:message>
      fnx:allowedRepTerms(<xsl:value-of select="$n/@name"/>) starting...
    </xsl:message>
    -->
    <xsl:variable name="result" as="xsd:string*">
      <xsl:choose>
        <xsl:when test="not($n)">
          <xsl:value-of select="()"/>
        </xsl:when>
        <xsl:when test="((local-name($n) = 'element') or (local-name($n) = 'attribute')) and not($n/@type)">
          <!-- use xsd:string for default -->
          <xsl:value-of select="('Text', 'Name', 'Number')"/>
        </xsl:when>
        <xsl:when test="$n/xsd:union">
          <xsl:value-of select="('Category', 'Code', 'Identifier', 'Number')"/>
        </xsl:when>
        <xsl:when test="$n/@type or $n/xsd:restriction or $n/xsd:simpleContent/*/@base or $n/*/@base">
          <xsl:variable name="basicType" select="($n/@type, $n/xsd:restriction/@base, $n/xsd:simpleContent/*/@base, $n/*/@base)[1]"/>
          <!--
          <xsl:message>
            fnx:allowedRepTerms(<xsl:value-of select="$n/@name"/>): basicType = <xsl:value-of select="$basicType"/>
          </xsl:message>
          -->
          <xsl:choose>
            <xsl:when test="$basicType = 'com:AmountType'">
              <xsl:value-of select="('Amount')"/>
            </xsl:when>
            <xsl:when test="$basicType = 'xsd:token'">
              <xsl:value-of select="('Category', 'Code', 'Identifier', 'Number')"/>
            </xsl:when>
            <xsl:when test="$basicType = 'xsd:boolean'">
              <xsl:value-of select="('Indicator')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'com:DateType') or ($basicType = 'xsd:date')">
              <xsl:value-of select="('Date')"/>
            </xsl:when>
            <xsl:when test="$basicType = 'com:MeasureType'">
              <xsl:value-of select="('Measure')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:positiveInteger')">
              <xsl:value-of select="('Number')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:decimal')">
              <xsl:value-of select="('Percent', 'Rate')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'com:QuantityType') or ($basicType = 'xsd:nonNegativeInteger')">
              <xsl:value-of select="('Quantity')"/>
            </xsl:when>
            <xsl:when test="$basicType = 'xsd:string'">
              <xsl:value-of select="('Text', 'Name', 'Number')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:time')">
              <xsl:value-of select="('Time')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:dateTime')">
              <xsl:value-of select="('DateTime')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:anyURI')">
              <xsl:value-of select="('URI')"/>
            </xsl:when>
            <xsl:when test="($basicType = 'xsd:anySimpleType')
                            or ($basicType = 'xsd:ID')
                            or ($basicType = 'xsd:IDREF')
                            or ($basicType = 'xsd:IDREFS')
                            ">
              <xsl:value-of select="('EXEMPT')"/>
            </xsl:when>
            <xsl:when test="$n/xsd:simpleContent/*/@base"> <!-- but matched none of the above -->
              <!-- <xsl:message>
                   fnx:allowedRepTerms(<xsl:value-of select="$n/@name"/>) recursing with $basicType = <xsl:value-of select="$basicType"/>  1...
                   </xsl:message> -->
              <xsl:value-of select="fnx:allowedRepTerms(fnx:get-xsd-definition(root($n), $basicType))"/>
            </xsl:when>
            <xsl:when test="$n/@type">
              <!-- <xsl:message>
                fnx:allowedRepTerms(<xsl:value-of select="$n/@name"/>) recursing 2...
              </xsl:message>-->
              <xsl:value-of select="fnx:allowedRepTerms(fnx:get-xsd-definition(root($n), $n/@type))"/>
            </xsl:when>
            <xsl:otherwise>()</xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>()</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!-- <xsl:message>
    fnx:allowedRepTerms(<xsl:value-of select="$n/@name"/>) returns <xsl:value-of select="$result"/>
    </xsl:message>-->
    <xsl:value-of select="$result"/>
  </xsl:function>

  <xsl:function name="fnx:usesAllowedRepTerm" as="xsd:boolean">
    <xsl:param name="repTerm" as="xs:string"/>    
    <xsl:param name="allowedRepTerms" as="xs:string?"/>
    <xsl:value-of select="contains(concat(' ', $allowedRepTerms, ' '), concat(' ', $repTerm, ' '))"/>
  </xsl:function>

  <!-- A Basic Component refers to a W3C Built-in Datatype, simple
       Type or complex Type with xsd:simpleContent definition.  It is
       represented by an element or an attribute.  For example,
       FirstName (string), PhoneNumberCategory
       (PhoneNumberCategoryType).-->
  <xsl:function name="fnx:isBasicComponent" as="xsd:boolean">
    <xsl:param name="n" as="node()"/>
    <!--<xsl:message>
        fnx:isBasicComponent(name=<xsl:value-of select="$n/@name"/> type=<xsl:value-of select="$n/@type"/>)
        </xsl:message>-->
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="not((local-name($n) = 'element') or (local-name($n) = 'attribute'))">
          <xsl:value-of select="false()"/>
        </xsl:when>
        <xsl:when test="not($n)">
          <xsl:value-of select="false()"/>
        </xsl:when>
        <xsl:when test="not($n/@type) and $n/*">
          <xsl:message>WARNING: <xsd:value-of select="$n/@name"/>:  XSD styles with nested type definitions are not currently supported.</xsl:message>
          <xsl:value-of select="false()"/>
        </xsl:when>
        <xsl:when test="not($n/@type)">
          <!-- default is xsd:string -->
          <xsl:value-of select="true()"/>
        </xsl:when>
        <xsl:when test="   $n/@type = 'com:AmountType'
                        or $n/@type = 'xsd:token'
                        or $n/@type = 'com:DateType'
                        or $n/@type = 'xsd:date'
                        or $n/@type = 'xsd:token'
                        or $n/@type = 'xsd:boolean'
                        or $n/@type = 'com:MeasureType'
                        or $n/@type = 'xsd:string'
                        or $n/@type = 'xsd:positiveInteger'
                        or $n/@type = 'xsd:decimal'
                        or $n/@type = 'com:QuantityType'
                        or $n/@type = 'xsd:nonNegativeInteger'
                        or $n/@type = 'xsd:time'
                        or $n/@type = 'xsd:dateTime'
                        or $n/@type = 'xsd:anyURI'
                        ">
          <xsl:value-of select="true()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="typeDef" select="fnx:get-xsd-definition(root($n), $n/@type)"/>
          <xsl:choose>
            <xsl:when test="$typeDef/xsd:simpleContent">
              <xsl:value-of select="true()"/>
            </xsl:when>
            <xsl:when test="local-name($typeDef) = 'simpleType'">
              <xsl:value-of select="true()"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="false()"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!--
    <xsl:message>
      fnx:isBasicComponent(<xsl:value-of select="$n/@name"/>) = <xsl:value-of select="$result"/>
    </xsl:message>
    -->
    <xsl:value-of select="$result"/>
  </xsl:function>


  <xsl:function name="fnx:makeCommaList" as="xsd:string">
    <xsl:param name="sList" as="xs:string*"/>
    <xsl:param name="orOrAnd" as="xs:string"/>

    <xsl:variable name="commaList" as="xs:string*">
      <xsl:for-each select="$sList">
        <xsl:choose>
          <xsl:when test="position() = 1">
            <xsl:value-of select="."/>
          </xsl:when>
          <xsl:when test="position() = last() and position() != 1">
            <xsl:text> </xsl:text>
            <xsl:value-of select="$orOrAnd"/>
            <xsl:text> </xsl:text>
            <xsl:value-of select="."/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>, </xsl:text>
            <xsl:value-of select="."/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>
    </xsl:variable>

    <xsl:value-of select="string-join($commaList, '')"/>

  </xsl:function>

  <xsl:function name="fnx:isTableRelated" as="xs:boolean">
    <xsl:param name="name" as="xs:string?"/>
    <xsl:value-of select="$name
                          and fnx:contains-i($name, 'table')
                          and not(matches($name, 'tablebody', 'i'))
                          and not(matches($name, 'tablegroup', 'i'))
                          and not(matches($name, 'tableimage', 'i'))
                          and not(matches($name, 'tabletitle', 'i'))
                          and not(matches($name, 'tabledata', 'i'))
                          and not(matches($name, 'tablecell', 'i'))
                          and not(matches($name, 'tabledatacell', 'i'))
                          and not(matches($name, 'tablefooter', 'i'))
                          and not(matches($name, 'tableheader', 'i'))
                          and not(matches($name, 'tableheadercell', 'i'))
                          and not(matches($name, 'tablerow', 'i'))
                          "/>
  </xsl:function>

  <xsl:function name="fnx:hasCorrectMixOfUnboundedChildrenToBeACollection" as="xs:boolean">
    <xsl:param name="complexType" as="element()?"/>
    <!--
         1. Has to be at least one xs:element[@maxOccurs='unbounded'].
         If there are multiple xs:elements, then all must be
         @maxOccurs='unbounded'
         or 
         2. Unbounded can be on xs:choice.
         or
         3. Any xs:element under an xs:choice can be unbounded.
    -->
    <xsl:value-of select="boolean($complexType)
                          and
                          (boolean($complexType/xs:choice[@maxOccurs = 'unbounded'])
                           or
                           boolean($complexType/xs:choice/xs:element[@maxOccurs = 'unbounded'])
                           or
                           boolean($complexType/*/xs:element[@maxOccurs = 'unbounded'])
                           or
                           boolean($complexType/*/xs:choice[@maxOccurs = 'unbounded'])
                          )"/>
<!--
                           (boolean($complexType/*/xs:element[@maxOccurs = 'unbounded'])
                            and
                            boolean(every $el in $complexType/*/xs:element satisfies $el/@maxOccurs = 'unbounded')
                           )
-->
  </xsl:function>


  <xsl:function name="fnx:isACollection" as="xs:boolean">
    <xsl:param name="e" as="element()?"/>

    <xsl:variable name="eIsACollection" as="xs:boolean">
      <xsl:choose>
        <xsl:when test="not($e)">
          <xsl:value-of select="false()"/>
        </xsl:when>
        <xsl:when test="$e/@mixed = 'true'">
          <xsl:value-of select="false()"/>
        </xsl:when>
        <xsl:when test="local-name($e) = 'element'">
          <xsl:value-of select="    fnx:hasCorrectMixOfUnboundedChildrenToBeACollection($e/xsd:complexType)
                                or ($e/@type and fnx:isACollection(fnx:get-xsd-definition(root($e), $e/@type)))"/>
        </xsl:when>
        <xsl:when test="local-name($e) = 'complexType'">
          <xsl:value-of select="fnx:hasCorrectMixOfUnboundedChildrenToBeACollection($e)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="false()"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!-- Useful debugging diagnostic:
    <xsl:choose>
      <xsl:when test="$eIsACollection">
        <xsl:message><xsl:value-of select="local-name($e)"/>: <xsl:value-of select="$e/@name"/> IS a collection</xsl:message>
      </xsl:when>
      <xsl:otherwise>
        <xsl:message><xsl:value-of select="local-name($e)"/>: <xsl:value-of select="$e/@name"/> is NOT a collection</xsl:message>
      </xsl:otherwise>
    </xsl:choose>
    -->
    <xsl:value-of select="$eIsACollection"/>
  </xsl:function>


<!--
  <pattern id="tp">
    <title>TESTpattern</title>
    <p>This is paragraph 1.</p>
    <p>This is paragraph 2.</p>
    <p class="note">note goes here</p>
    <p class="description">description goes here</p>
    <p class="ruleDescription">rule description goes here</p>
    <p class="autoNotes">auto notes go here</p>
    <rule context="/xsd:schema" role="INFO">
      <report test="." flag="MANUAL">This is the INFO message from TESTpattern.</report>
    </rule>
  </pattern>
-->

  <pattern>
    <title>GD-01</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-02</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
         <assert test="xsd:schema" flag="AUTO" role="ERROR">The root element is not an xsd:schema element.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-03</title>
    <rule context="/">
        <let name="this-as-text" value="unparsed-text(document-uri(/))"/>
        <let name="first-line" value="replace($this-as-text,'^(.*?)[\n\r]+.*$','$1','s')"/>
        <assert test="matches($first-line,'encoding\s*=\s*&quot;(UTF|utf)-8&quot;')" flag="AUTO" role="ERROR">The XML declaration (first line) of the document does not indicate an encoding of UTF-8.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-04</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-05</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-06</title>
    <rule context="  xsd:complexType[@name]
                   | xsd:simpleType[@name]
                   | xsd:element[@name]
                   | xsd:attribute[@name]">
      <assert test="fnx:is-exception('GD-06') or
                    matches(@name, '^[a-zA-Z0-9]+$')"
              flag="AUTO"
              role="ERROR">
        The name of
        <value-of select="concat(local-name(.), ' ', @name)"/>
        contains invalid characters.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-07</title>
    <rule context="  xsd:complexType[@name]
                   | xsd:simpleType[@name]
                   | xsd:element[@name]
                   | xsd:attribute[@name]">
      <assert test="fnx:is-exception('GD-07') or string-length(@name) &lt;= 35"
              flag="AUTO"
              role="WARNING">
        The length of the
        <value-of select="local-name(.)"/>
        named
        <value-of select="@name"/>
        is
        <value-of select="string-length(@name)"/>.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-08</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <!-- Note: All UCC rules are only partially checked.  Without
       consulting a full dictionary and making assumption about word
       boundaries, we cannot check interior capitalizations.  Examples
       we catch: 'badElementName' and 'BADELEMENTNAME'.  Examples we
       miss: 'BadelementName' and 'BadElementname'. -->
  <pattern>
    <title>GD-09</title>
    <rule context="xsd:element[@name]">
      <assert test="fnx:is-exception('GD-09') or
                    matches(@name,'^[A-Z]')" flag="AUTO" role="ERROR">
        The element name <value-of select="@name"/> does not start with an upper-case letter.
      </assert>
      <assert test="fnx:is-exception('GD-09') or
                    not(matches(@name,'^[A-Z][A-Z][A-Z]+$')) or
                    (@name = $approvedAcronyms/AAs/AA/short)" flag="AUTO" role="ERROR">
        The element name <value-of select="@name"/> contains all
        upper-case letters instead of using camel case.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-10</title>
    <rule context="xsd:complexType[@name] | xsd:simpleType[@name]">
      <assert test="fnx:is-exception('GD-10')
                    or matches(@name,'^[A-Z]')" flag="AUTO" role="ERROR">
        The <value-of select="local-name()"/> name <value-of
        select="@name"/> does not start with an upper-case letter.
      </assert>
      <assert test="fnx:is-exception('GD-10') or
                    not(matches(@name,'^[A-X][A-Z]+$'))"
              flag="AUTO" role="ERROR">
        The <value-of select="local-name()"/> name <value-of
        select="@name"/> contains all upper-case letters instead of
        using camel case.
      </assert>
      <assert test="fnx:is-exception('GD-10') or
                    ends-with(@name,'Type')" flag="AUTO" role="ERROR">
        The <value-of select="local-name()"/> name <value-of
        select="@name"/> does not end with Type.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-11</title>
    <rule context="xsd:attribute[@name]">
      <assert test="fnx:is-exception('GD-11') or
                    matches(@name,'^[a-z]')" flag="AUTO" role="ERROR">
        Attribute name <value-of select="@name"/> does not start with
        a lower-case letter.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-12</title>
    <rule context="xsd:element[@name]">
      <let name="name" value="@name"/>
      <let name="aaEntry" value="$approvedAcronyms/AAs/AA[contains(replace(lower-case($name), ' ', ''), replace(lower-case(long), ' ', ''))]"/>
         <assert test="fnx:is-exception('GD-12') or not($aaEntry)" role="ERROR">
            <value-of select="replace($aaEntry/long, ' ', '')"/> in <value-of select="@name"/> should be abbreviated as <value-of select="$aaEntry/short"/>.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-13</title>
    <!--
        Automation concerns:

        (1) Single-letter and double-letter abbreviations cannot be
        detected apart from regular single and double letters within
        words.

        (2) Embedded multi-letter sequences can appear naturally
        within words without being abbreviations.

        Assumption to automate:

        (1) Don’t automate single-letter and double-letter
        abbreviations.

        (2) Accept some false-positive reports.
    -->
    <rule context="xsd:attribute[@name]">
      <let name="improperEmbeddedAbbreviations" value="distinct-values(fnx:find-improper-embedded-abbreviation-in-attribute(@name))"/>
      <let name="ieaWithoutEmptyString" value="remove($improperEmbeddedAbbreviations, index-of($improperEmbeddedAbbreviations, '')[1])"/>
      <assert test="fnx:is-exception('GD-13') or
                    empty($ieaWithoutEmptyString)" role="ERROR">
        Attribute <value-of select="@name"/> contains these improperly capitalized abbreviations: <value-of select="string-join($ieaWithoutEmptyString, ',')"/>.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-14</title>
    <rule context="xsd:element[@name]|xsd:complexType[@name]|xsd:simpleType[@name]">
         <let name="improperEmbeddedAbbreviations" value="distinct-values(fnx:find-improper-embedded-abbreviation-in-element-or-type(@name))"/>
         <let name="ieaWithoutEmptyString" value="remove($improperEmbeddedAbbreviations, index-of($improperEmbeddedAbbreviations, '')[1])"/>
         <assert test="fnx:is-exception('GD-14') or
                    empty($ieaWithoutEmptyString)" role="ERROR">
        Component <value-of select="@name"/> contains these improperly capitalized abbreviations: <value-of select="string-join($ieaWithoutEmptyString, ',')"/>.</assert>
      </rule>
  </pattern>
  <pattern>
    <title>GD-15</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-16</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-17</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-18</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-19</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-20</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-21</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-22</title>
    <rule context="xsd:element[@name] | xsd:attribute[@name] | xsd:simpleType[@name] | xsd:complexType[@name]">
      <let name="name" value="@name"/>
      <let name="nameWithRepeatedRepTerm" value="string-join(
                                                 (for $repTerm in $kRepresentationTerms
                                                 return 
                                                 if (contains(lower-case($name),
                                                 lower-case(concat($repTerm, $repTerm))))
                                                 then $repTerm
                                                 else ''),
                                                 '')"/>
      <assert test="fnx:is-exception('GD-22') or
                    $nameWithRepeatedRepTerm = ''" role="ERROR">
        <value-of select="$name"/> contains a repeated representation term (<value-of select="$nameWithRepeatedRepTerm"/>).
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-23</title>
    <rule context="  xsd:element[@name]
                   | xsd:attribute[@name]">

      <let name="name" value="@name"/>
      <let name="repTerm" value="fnx:getRepTerm($name)"/>
      <let name="allowedRepTerms" value="fnx:allowedRepTerms(.)"/>

      <assert test="fnx:is-exception('GD-23')
                    or not(fnx:isBasicComponent(.))
                    or fnx:usesAllowedRepTerm($repTerm, $allowedRepTerms)"
              role="ERROR">
        <value-of select="$name"/>
        must have
        <value-of select="if ($allowedRepTerms = '') then ' no representation term, ' else concat(fnx:makeCommaList(tokenize($allowedRepTerms, ' '), 'or'), ' as a representation term,')"/>
        <value-of select="if ($repTerm = '') then ' but it has none.' else concat(' not ', $repTerm, '.')"/>
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-24</title>
    <rule context="xsd:element[@name] | xsd:attribute[@name] | xsd:simpleType[@name] | xsd:complexType[@name]">
      <let name="thisName" value="@name"/>
      <let name="componentName" value="local-name()"/>
      <let name="count" value="count(//*[@name = $thisName])"/>
      <assert test="fnx:is-exception('GD-24') or
                    $count = 1" role="ERROR">
        Name <value-of select="$thisName"/> is not unique.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-25</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-26</title>

    <!-- A collection must be named with Bag suffix -->

    <rule context="/xsd:schema/xsd:element[@name and fnx:isACollection(.)]">
      <assert test="fnx:is-exception('GD-26') or
                    .[ends-with(@name, 'Bag')]" role="WARNING">
        The '<value-of select="@name"/>' element is a collection, but its name lacks a 'Bag' suffix.
      </assert>
    </rule>

    <rule context="/xsd:schema/xsd:complexType[@name and fnx:isACollection(.)]">
      <assert test="fnx:is-exception('GD-26') or
                    .[ends-with(@name, 'BagType')]" role="WARNING">
        The '<value-of select="@name"/>' complexType is a collection, but its name lacks a 'BagType' suffix.
      </assert>
    </rule>

    <rule context="/xsd:schema/xsd:simpleType[@name and fnx:isACollection(.)]">
      <assert test="fnx:is-exception('GD-26') or
                    .[ends-with(@name, 'BagType')]" role="WARNING">
        The '<value-of select="@name"/>' simpleType is a collection, but its name lacks a 'BagType' suffix.
      </assert>
    </rule>

    <!-- A non-collection must not be named with Bag suffix -->

    <rule context="/xsd:schema/xsd:element[@name and not(fnx:isACollection(.))]">
      <assert test="fnx:is-exception('GD-26') or
                    .[not(ends-with(@name, 'Bag'))]" role="WARNING">
        The '<value-of select="@name"/>' element is not a collection, but its name has a 'Bag' suffix.
      </assert>
    </rule>

    <rule context="/xsd:schema/xsd:complexType[@name and not(fnx:isACollection(.))]">
      <assert test="fnx:is-exception('GD-26') or
                    .[not(ends-with(@name, 'BagType'))]" role="WARNING">
        The '<value-of select="@name"/>' complexType is not a collection, but its name has a 'BagType' suffix.
      </assert>
    </rule>

    <rule context="/xsd:schema/xsd:simpleType[@name and not(fnx:isACollection(.))]">
      <assert test="fnx:is-exception('GD-26') or
                    .[not(ends-with(@name, 'BagType'))]" role="WARNING">
        The '<value-of select="@name"/>' simpleType is not a collection, but its name has a 'BagType' suffix.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-27</title>
    <rule context="xsd:complexType[@name]|xsd:simpleType[@name]|xsd:element[@name]|xsd:attribute[@name]">
         <let name="nameparts" value="fnx:words-from-name(@name)"/>
         <let name="connectingWordsInName" value="string-join($connectingWords[. = $nameparts], ', ')"/>
         <assert test="fnx:is-exception('GD-27') or $connectingWordsInName=''" flag="AUTO" role="WARNING">
        Name of <value-of select="concat(local-name(.),' ',@name)"/>
        contains the connecting word(s) <value-of select="$connectingWordsInName"/>.
      </assert>
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-28</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>GD-29</title>
      <rule context="xsd:element[@name]|xsd:complexType">
         <let name="name" value="@name"/>
         <let name="legalReferenceMatches" value="$legalReferencePrefixes[matches($name,concat(.,'\d'))]"/>
         <assert test="fnx:is-exception('GD-29') or
                    not($legalReferenceMatches)" flag="AUTO" role="ERROR">
        The name <value-of select="@name"/> contains an article or rule number (<value-of select="$legalReferenceMatches"/> followed by a digit).
      </assert>
         <report test="false()" flag="MANUAL"/>
      </rule>
  </pattern>
  <pattern>
    <title>GD-30</title>
    <rule context="/">
         <assert test="fnx:is-exception('GD-30') or
                    matches($localFile,'^[a-zA-Z0-9_.]+$')" flag="AUTO" role="ERROR">
        The file name <value-of select="$localFile"/> contains invalid
        characters.
      </assert>
      </rule>
  </pattern>
  <pattern>
    <title>GD-31</title>
    <!-- Why is there nothing in GD-31 about the version in the file
         name matching schema/@version?  GD-32 addresses it but for
         drafts.
    -->
    <rule context="/">
         <assert test="fnx:is-exception('GD-31') or
                    matches($localFile, 'draft', 'i')
                    or matches($localFile, '_D', 'i')
                    or matches($localFile,'^[a-zA-Z0-9]+(_V[0-9]+_[0-9]+)?\.[a-zA-Z0-9]+$')" flag="AUTO" role="ERROR">
        The file name <value-of select="$localFile"/> does not follow the required format &#x003C;ComponentName&#x003E;[_V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E;].&#x003C;FileExtension&#x003E;
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>GD-32</title>
    <rule context="/xsd:schema">
      <let name="xsdNameIsOk"
           value="(not(matches($localFile, 'draft', 'i'))
                    and not(matches($localFile, '_D[0-9]+', 'i')))
                    or
                    matches($localFile,'^[a-zA-Z0-9]+(_V[0-9]+_[0-9]+)?_D[0-9]+\.[a-zA-Z0-9]+$')"/>
      <let name="fileNameVersion"
           value="fnx:xsdVersionFromFileName($localFile)"
           />
      <let name="attVersion"
           value="@version"
           />
      <let name="nameVersionIsConsistentWithAttributeVersion"
           value="not($fileNameVersion) or $fileNameVersion = $attVersion"
           />

      <assert test="fnx:is-exception('GD-32') or
                    $xsdNameIsOk" flag="AUTO">
        The file name <value-of select="$localFile"/>  does not follow the required format for drafts: &#x003C;ComponentName&#x003E;[_V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E;]_D&#x003C;RevisionNumber&#x003E;.&#x003C;FileExtension&#x003E;
      </assert>

      <assert test="fnx:is-exception('GD-32') or
                    not($xsdNameIsOk) or $nameVersionIsConsistentWithAttributeVersion" flag="AUTO">
        The version in XSD name <value-of select="$localFile"/> (<value-of select="$fileNameVersion"/>) is inconsistent with the version given by schema/@version (<value-of select="$attVersion"/>).
      </assert>

    </rule>
  </pattern>
  <pattern>
    <title>SD-01</title>
    <!-- SD-01: "Patent Component schema modules, Trademark Component
         schema modules, and Design Component schema modules MUST use
         the xsd:import construct to reference Common Component schema
         modules."

         That xsd:import is required to use components from another
         namespace is already implied by XML Schema itself.  Repeating
         this as a guideline adds nothing on its own and should
         probably be removed as a ST.96 rule.  Worse, it appears to
         require that Common be imported even when not used.  It's
         confusing.  Per meeting with Narith and Sunil, implement this
         check as follows: If a Patent, Trademark, or Design component
         module indicates an intention to use Common by declaring its
         namespace prefix on xsd:schema, then make sure that there is
         actually an xsd:import of Common.  Warning when a Common
         namespace prefix is present without an associated xsd:import
         of Common will at least catch dangling namespace prefixes
         declarations.
    -->
    <rule context="/xsd:schema[(@targetNamespace = $kPatentNamespace) and (fnx:getNSPrefix(., $kCommonNamespace) = 'com')]">
      <assert test="fnx:is-exception('SD-01') or
                    xsd:import[@namespace = $kCommonNamespace]" role="WARNING">
        XSDs in the patent namespace that have declared a common namespace prefix must use xsd:import to reference common component schema modules.
      </assert>
    </rule>
    <rule context="/xsd:schema[(@targetNamespace = $kTrademarkNamespace) and (fnx:getNSPrefix(., $kCommonNamespace) = 'com')]">
      <assert test="fnx:is-exception('SD-01') or
                    xsd:import[@namespace = $kCommonNamespace]" role="WARNING">
        XSDs in the trademark namespace that have declared a common namespace prefix must use xsd:import to reference common component schema modules.
      </assert>
    </rule>
    <rule context="/xsd:schema[(@targetNamespace = $kDesignNamespace) and (fnx:getNSPrefix(., $kCommonNamespace) = 'com')]">
      <assert test="fnx:is-exception('SD-01') or 
                    xsd:import[@namespace = $kCommonNamespace]" role="WARNING">
        XSDs in the design namespace that have declared a common namespace prefix must use xsd:import to reference common component schema modules.
      </assert>
    </rule>
<rule context="/xsd:schema[(@targetNamespace = $kIPONonCommonNamespaces) and (fnx:getNSPrefix(., $kIPOCommonNamespace) = $kIPOCommonNamespacePrefix)]">
         <assert test="fnx:is-exception('SD-01') or
                    xsd:import[@namespace = $kIPOCommonNamespace]" role="WARNING">
        XSDs in an IPO namespace that have declared a common namespace prefix must use xsd:import to reference IPO common component schema modules.
      </assert>
      </rule>
    </pattern>
   <pattern>
    <title>SD-02</title>
    <rule context="/xsd:schema[@targetNamespace = $kCommonNamespace or @targetNamespace = $kIPOCommonNamespace]">
         <let name="illegalImport" value="xsd:import
                  [  not(@namespace = ($kMathMLNamespace, $kOASISTableNamespace, $kCommonNamespace)) ]"/>
         <assert test="fnx:is-exception('SD-02') or
                    not($illegalImport)">
        XSDs in the <value-of select="@targetNamespace"/> namespace
        may not refer to XSDs in the following namespace(s):
        <value-of select="string-join(distinct-values($illegalImport/@namespace),', ')"/>.
      </assert>
      </rule>
   </pattern>
   <pattern>
    <title>SD-03</title>
    <rule context="/xsd:schema[@targetNamespace = ($kNonCommonNamespaces,$kIPONonCommonNamespaces)]">
         <!-- there is no need to check if it is the same as the target namespace because
              this is disallowed by XSD -->
        <let name="tns" value="@targetNamespace"/>
         <let name="tns-component" value="$namespaces[@ns=$tns]/@component"/>
         <let name="bad-namespaces" value="$namespaces[@component!='common' and @component!=$tns-component]/@ns"/>
         <let name="illegalImport" value="xsd:import[@namespace = $bad-namespaces]"/>
         <assert test="fnx:is-exception('SD-03') or
                    not($illegalImport)">
        XSDs in the <value-of select="@targetNamespace"/> namespace may not refer to XSDs in the following namespace(s):
        <value-of select="string-join(distinct-values($illegalImport/@namespace),', ')"/>. 
      </assert>
      </rule>
   </pattern>
   <pattern>
    <title>SD-04</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-05</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-06</title>
      <rule context="xsd:element | xsd:attribute | xsd:simpleType | xsd:complexType">
         <assert test="fnx:is-exception('SD-06') or
                    not(@name) or parent::xsd:schema" flag="AUTO" role="ERROR">
            <value-of select="@name"/> (<value-of select="name()"/>) is declared locally.</assert>
      </rule>
   </pattern>
   <pattern>
    <title>SD-07</title>
    <rule context="xsd:redefine">
         <assert test="fnx:is-exception('SD-07') or
                    false()" flag="AUTO" role="ERROR">
        xsd:redefine must not be used.
      </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-08</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-08') or
                    @targetNamespace">
        xsd:schema has no targetNamespace attribute.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-09</title>
    <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-09') or
                    @targetNamespace = $namespaces/@ns" flag="AUTO" role="ERROR">
        Target namespace <value-of select="@targetNamespace"/> is not one of the baseline namespaces.
      </assert>
         <report test="false()" flag="MANUAL"/>
      </rule>
      <rule context="xsd:import">
         <assert test="fnx:is-exception('SD-09') or
                    @namespace = $namespaces/@ns" flag="AUTO" role="ERROR">
        Imported namespace <value-of select="@namespace"/> is not one of the baseline namespaces.
      </assert>
      </rule>
   </pattern>
  <pattern>
    <title>SD-10</title>
    <rule context="/*">
      <assert test="fnx:is-exception('SD-10') or
                    namespace::node()[.='http://www.w3.org/2001/XMLSchema']" flag="AUTO" role="ERROR">
        Schema does not declare the XML Schema namespace
        (http://www.w3.org/2001/XMLSchema).
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-11</title>
    <!-- SD-11 ("Schemas MUST use namespace qualifications for all W3C
         Schema constructs.") is already implied by XML Schema
         constraints in general.  It adds nothing on its own and
         should probably be removed as a ST.96 rule.  Perhaps it is
         trying to say that Schemas must use namespace prefixes for
         all W3C Schema constructs?  Yet per
         http://www.w3.org/TR/REC-xml-names/#dt-qualname [Definition:
         A qualified name is a name subject to namespace
         interpretation [...] Syntactically, they are either prefixed
         names or unprefixed names.  Will assume intent of SD-11 is to
         insist that *some* namespace prefix MUST be used to reference
         XSD constructs, whereas SD-15 says that specifically "xsd"
         SHOULD be used. ]
    -->
    <rule context="/xsd:schema">
      <let name="xsdAssertionError"
           value="fnx:assertAnyNSPrefix(., $kXMLSchemaNamespace)"/>

      <assert test="fnx:is-exception('SD-11') or
                    $xsdAssertionError = ''" flag="AUTO" role="ERROR">
        <value-of select="$xsdAssertionError"/>
      </assert>

    </rule>
  </pattern>
  <xsl:function name="fnx:isURI" as="xs:boolean">
    <xsl:param name="targetNamespace" as="xs:string?"/>
    <!-- Could be more thorough here, but this should adequately cover
         common cases. -->
    <xsl:value-of select="$targetNamespace and
                          (starts-with($targetNamespace, 'http:')
                          or starts-with($targetNamespace, 'https:')
                          or starts-with($targetNamespace, 'urn:'))
                          "/>
  </xsl:function>
  <pattern>
    <title>SD-12</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-12') or
                    fnx:isURI(@targetNamespace)" role="ERROR">
        @targetNamespace=<value-of select="@targetNamespace"/> is not a valid URI.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-13</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-13') or
                    @elementFormDefault = 'qualified'" role="ERROR">
        @elementFormDefault must be qualified.
      </assert>
      <assert test="fnx:is-exception('SD-13') or
                    @attributeFormDefault = 'qualified'" role="ERROR">
        @attributeFormDefault must be qualified.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-14</title>
    <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-14')
                     or not($inComponentSubdirName = ('Common', 'Patent', 'Trademark', 'Design'))
                     or ($inComponentSubdirName = 'Common' and @targetNamespace = $kCommonNamespace)
                     or ($inComponentSubdirName = 'Patent' and @targetNamespace = $kPatentNamespace)
                     or ($inComponentSubdirName = 'Trademark' and @targetNamespace = $kTrademarkNamespace)
                     or ($inComponentSubdirName = 'Design' and @targetNamespace = $kDesignNamespace)
                    " flag="AUTO" role="ERROR">
        Namespace <value-of select="@targetNamespace"/> does not follow the proper form.
      </assert>
      
      <let name="tns" value="@targetNamespace"/>
      <assert test="fnx:is-exception('SD-14') 
                  or not($inComponentSubdirName = ('USCommon', 'USPatent', 'USTrademark', 'USDesign', 'USCorporate', 'USDissemination', 'USLegal'))
                  or ($namespaces[@subdir = $inComponentSubdirName and @ns=$tns])" flag="AUTO">
        Namespace <value-of select="@targetNamespace"/> does not follow the proper form. Expected namespace <value-of select="$namespaces[@subdir=$inComponentSubdirName]/@ns"/> based on subdirectory name <value-of select="$inComponentSubdirName"/>.
      </assert>
      </rule>
  </pattern>

  <xsl:function name="fnx:getNSPrefix" as="xs:string">
    <xsl:param name="n" as="node()"/>
    <xsl:param name="nsName" as="xs:string"/>
    <xsl:variable name="nsNode" select="$n/namespace::node()[.=$nsName]"/>
    <xsl:value-of select="local-name($nsNode)"/>
  </xsl:function>

  <xsl:function name="fnx:assertSpecificNSPrefix" as="xs:string?">
    <xsl:param name="n" as="node()"/>
    <xsl:param name="nsName" as="xs:string"/>
    <xsl:param name="shouldBePrefix" as="xs:string"/>
    <xsl:variable name="prefix" select="fnx:getNSPrefix($n, $nsName)"/>
    <xsl:choose>
      <xsl:when test="$prefix = '' or $prefix = $shouldBePrefix">
        <xsl:sequence select="()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat($shouldBePrefix, ': prefix (not ', $prefix, ':) must be used for the ', $nsName, ' namespace.')"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>

  <xsl:function name="fnx:assertAnyNSPrefix" as="xs:string">
    <xsl:param name="n" as="node()"/>
    <xsl:param name="nsName" as="xs:string"/>

    <xsl:variable name="prefix" select="fnx:getNSPrefix($n, $nsName)"/>
    <xsl:choose>
      <xsl:when test="$prefix = ''">
        <xsl:value-of select="concat('A namespace prefix must be used for the ', $nsName, ' namespace.')"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="''"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>

  <xsl:function name="fnx:contains-i" as="xs:boolean">
    <xsl:param name="s" as="xs:string"/>
    <xsl:param name="subs" as="xs:string"/>
    <xsl:value-of select="contains(lower-case($s), lower-case($subs))"/>
  </xsl:function>

  <xsl:function name="fnx:find-i" as="xs:boolean">
    <xsl:param name="s" as="xs:string"/>
    <xsl:param name="subs" as="xs:string"/>
    <xsl:value-of select="contains(lower-case($s), lower-case($subs))"/>
  </xsl:function>

  <xsl:function name="fnx:get-xsd-definition" as="node()?">
    <xsl:param name="schemaRoot" as="node()"/>
    <xsl:param name="typeName" as="xsd:string"/>

    <!--<xsl:message>get-xsd-definition(<xsl:sequence select="$typeName"/>)</xsl:message>-->

    <xsl:variable name="typeNameNoPrefix" select="if (contains($typeName, ':')) then substring-after($typeName, ':') else $typeName"/>
    
    <xsl:variable name="nsPrefix" as="xsd:string" select="substring-before($typeName, ':')"/>
    
    <xsl:variable name="nsName" as="xsd:string">
      <xsl:choose>
        <xsl:when test="$nsPrefix = '' and $schemaRoot/xsd:schema/@targetNamespace">
          <xsl:value-of select="$schemaRoot/xsd:schema/@targetNamespace"/>
        </xsl:when>
        <xsl:when test="$nsPrefix = 'com'"> <!-- Fixes bug related to missing com namespace in non-common components --> 
          <xsl:value-of select="$kCommonNamespace"/>
        </xsl:when>
        <xsl:when test="$nsPrefix = ''">
          <xsl:value-of select="'DEFAULT'"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$prefixTable/d:prefixBinding[@nsPrefix = $nsPrefix]/@nsName"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    	
    <xsl:variable name="result" select="$lookupTable/d:entry[@ns=$nsName]/*[@name=$typeNameNoPrefix][1]"/>

    <!--
    <xsl:message>$result=|<xsl:sequence select="$result"/>|</xsl:message>
    -->
    <xsl:sequence select="$result"/>
  </xsl:function>

  <xsl:function name="fnx:starts-with-ST3-code" as="xsd:boolean">
    <xsl:param name="s" as="xsd:string"/>

    <xsl:value-of select="substring($s,1,2) = $st3Codes"/>
  </xsl:function>

  <xsl:function name="fnx:has-fixed-attribute" as="xsd:boolean">
    <xsl:param name="el" as="element()"/>
    <xsl:param name="attName" as="xsd:string"/>
    <xsl:param name="use" as="xsd:string"/>
    <xsl:param name="fixedRegex" as="xsd:string"/>

    <xsl:variable name="def" select="fnx:get-xsd-definition(root($el), $el/@type)"/>
    <xsl:variable name="attdef1" select="fnx:get-attributes-from-type-recursive($def,$attName)"/>
      <xsl:variable name="attdef2" select="if ($attdef1/@name) then $attdef1 else fnx:get-xsd-definition(root($el), $attdef1/@ref)"/>
      <xsl:value-of select="boolean($attdef1[@use=$use or (not(@use) and $use = 'optional')] and ($attdef1|$attdef2)[matches(@fixed, $fixedRegex)])"/>
   </xsl:function>
<xsl:function name="fnx:get-attributes-from-type-recursive" as="element()*">
      <xsl:param name="ty" as="element()?"/>
      <xsl:param name="attName" as="xsd:string"/>
      <xsl:variable name="local-def" select="$ty//xsd:attribute[(@name=$attName or ends-with(@ref,$attName))]"/>
      <xsl:choose>
         <xsl:when test="$local-def">
            <xsl:sequence select="$local-def"/>
         </xsl:when>
         <xsl:otherwise>
          <xsl:variable name="extension" select="$ty/(xsd:complexContent|xsd:simpleContent)/xsd:extension"/>
          <xsl:choose>
             <xsl:when test="$extension">
               <xsl:variable name="base-def-ext" select="fnx:get-xsd-definition(root($ty), $extension/@base)"/>
              <xsl:sequence select="fnx:get-attributes-from-type-recursive($base-def-ext,$attName)"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:variable name="restriction" select="$ty/(xsd:complexContent|xsd:simpleContent)/xsd:restriction"/>
             <xsl:if test="$restriction">
               <xsl:variable name="base-def-res" select="fnx:get-xsd-definition(root($ty), $restriction/@base)"/>
              <xsl:sequence select="fnx:get-attributes-from-type-recursive($base-def-res,$attName)"/>
             </xsl:if>
           </xsl:otherwise>
          </xsl:choose>
         </xsl:otherwise>
      </xsl:choose>
   </xsl:function>
  <xsl:function name="fnx:uses-type" as="xsd:boolean">
    <!-- Rough estimation of whether the given XSD node, n, uses the
         given type, typeName.  Reproducing all of XSD type semantics
         here would be a monumental task; the goal is just to cover
         many of the common cases.
    -->
    <xsl:param name="n" as="node()?"/>
    <xsl:param name="usesTypeName" as="xsd:string"/>
    <xsl:param name="depth" as="xsd:integer"/>

    <xsl:choose>
      <xsl:when test="$depth &lt; 6"> <!-- performance concession and loop avoidance -->
        <xsl:variable name="result">
          <xsl:choose>
            <xsl:when test="empty($n)">
              <xsl:value-of select="false()"/>
            </xsl:when>
            <xsl:when test="local-name($n) = 'extension'">
              <xsl:value-of select="$n/@base = $usesTypeName
                                    or
                                    fnx:uses-type(fnx:get-xsd-definition(root($n), $n/@base), $usesTypeName, $depth+1)
                                    or
                                    fnx:uses-type($n/*[1], $usesTypeName, $depth+1)
                                    "/>
            </xsl:when>
            <xsl:when test="local-name($n) = 'element'">
              <xsl:choose>
                <xsl:when test="$n/@type and $n[@type = $usesTypeName]">
                  <xsl:value-of select="true()"/>
                </xsl:when>
                <xsl:when test="$n/@type">
                  <xsl:value-of select="fnx:uses-type(fnx:get-xsd-definition(root($n), $n/@type), $usesTypeName, $depth+1)"/>
                </xsl:when>
                <xsl:when test="$n/@ref and $n[@ref = $usesTypeName]">
                  <xsl:value-of select="true()"/>
                </xsl:when>
                <xsl:when test="$n/@ref">
                  <xsl:value-of select="fnx:uses-type(fnx:get-xsd-definition(root($n), $n/@ref), $usesTypeName, $depth+1)"/>
                </xsl:when>
                <xsl:otherwise><xsl:value-of select="false()"/></xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:when test="local-name($n) = 'attribute'">
              <!-- TBD -->
              <xsl:value-of select="false()"/>
            </xsl:when>
            <xsl:when test="local-name($n) = 'complexType'">
              <xsl:choose>
                <!-- For now, if type is used anywhere beneath the
                     complexType by an xsd:element or xsd:extension/@base,
                     we'll say it's good enough for us. Could handle
                     xsd:sequence, xsd:choice, etc differently. -->
                <xsl:when test="$n/@name = $usesTypeName"><xsl:value-of select="true()"/></xsl:when>
                <xsl:when test="some $child in $n//(xsd:element|xsd:extension) satisfies fnx:uses-type($child, $usesTypeName, $depth+1)"><xsl:value-of select="true()"/></xsl:when>
                <xsl:otherwise><xsl:value-of select="false()"/></xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:when test="local-name($n) = 'simpleType'">
              <xsl:choose>
                <xsl:when test="$n/xsd:union">
                  <xsl:value-of select="contains($n/xsd:union/@memberTypes, $usesTypeName)"/>
                </xsl:when>
                <xsl:otherwise><xsl:value-of select="false()"/></xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:otherwise><xsl:value-of select="false()"/></xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <!--<xsl:message>*** fnx:uses-type() is returning <xsl:value-of select="$result"/> ***</xsl:message>-->
        <xsl:value-of select="$result"/>
      </xsl:when>
      <xsl:otherwise><xsl:value-of select="false()"/></xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  <pattern>
      <title>SD-15</title>
      <rule context="/xsd:schema">
         <let name="schema" value="."/>
         <let name="errors" value="$namespaces/fnx:assertSpecificNSPrefix($schema, @ns, @ns-prefix)"/>
         <assert test="fnx:is-exception('SD-15') or empty($errors)" flag="AUTO" role="WARNING">
            <value-of select="string-join($errors,', ')"/>
         </assert>
      </rule>
   </pattern>
  <pattern>
    <title>SD-16</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-16') or
                    @targetNamespace">
        xsd:schema has no targetNamespace attribute.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-17</title>
    <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-17') or
                    @targetNamespace != $kXMLSchemaNamespace">
        @targetNamespace must not equal <value-of select="$kXMLSchemaNamespace"/>.
         </assert>
         <let name="tns" value="@targetNamespace"/>
         <assert test="fnx:is-exception('SD-17') or
                       namespace::node()[.=$tns]">
           @targetNamespace must be one of the declared namespaces. 
         </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-18</title>
     <!-- Ascertain Common vs Patent vs Trademark vs Design based upon
         name of containing subdirectory -->
      <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-18') or
                    $inSubdirName != 'Common' or @targetNamespace = $kCommonNamespace" flag="AUTO">
        For XSD found in the Common subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kCommonNamespace"/>
         </assert>
            <assert test="fnx:is-exception('SD-18') or
                       $inSubdirName != 'USCommon' or @targetNamespace = $kIPOCommonNamespace" flag="AUTO">
           For XSD found in the USCommon subdirectory, @targetNamespace
           (<value-of select="@targetNamespace"/>) must equal <value-of select="$kIPOCommonNamespace"/>
            </assert>
      </rule>
   </pattern>
  <pattern>
    <title>SD-19</title>
    <!-- Ascertain Common vs Patent vs Trademark vs Design based upon
         name of containing subdirectory -->
      <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-19') or
                    $inComponentSubdirName != 'Patent' or @targetNamespace = $kPatentNamespace" flag="AUTO">
        For XSD found in the Patent subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kPatentNamespace"/>
         </assert>
         <assert test="fnx:is-exception('SD-19') or
                    $inComponentSubdirName != 'USPatent' or @targetNamespace = $kIPOPatentNamespace" flag="AUTO">
        For XSD found in the USPatent subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kIPOPatentNamespace"/>
         </assert>
      </rule>
   </pattern>
  <pattern>
    <title>SD-20</title>
    <!-- Ascertain Common vs Patent vs Trademark vs Design based upon
         name of containing subdirectory -->
      <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-20') or
                    $inComponentSubdirName != 'Trademark' or @targetNamespace = $kTrademarkNamespace" flag="AUTO">
        For XSD found in the Trademark subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kTrademarkNamespace"/>
         </assert>
         <assert test="fnx:is-exception('SD-20') or
                    $inComponentSubdirName != 'USTrademark' or @targetNamespace = $kIPOTrademarkNamespace" flag="AUTO">
        For XSD found in the USTrademark subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kIPOTrademarkNamespace"/>
         </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-21</title>
    <!-- Ascertain Common vs Patent vs Trademark vs Design based upon
         name of containing subdirectory -->
      <rule context="/xsd:schema">
         <assert test="fnx:is-exception('SD-21') or
                    $inComponentSubdirName != 'Design' or @targetNamespace = $kDesignNamespace" flag="AUTO">
        For XSD found in the Design subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kDesignNamespace"/>
         </assert>
         <assert test="fnx:is-exception('SD-21') or
                    $inComponentSubdirName != 'USDesign' or @targetNamespace = $kIPODesignNamespace" flag="AUTO">
        For XSD found in the USDesign subdirectory, @targetNamespace
        (<value-of select="@targetNamespace"/>) must equal <value-of select="$kIPODesignNamespace"/>
         </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-22</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-22') or
                    @targetNamespace">
        xsd:schema has no targetNamespace attribute.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-23</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-24</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-25</title>
    <rule context="/">
      <report test="false()" flag="DELETED"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-26</title>
    <rule context="/">
      <report test="false()" flag="DELETED"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-27</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-28</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-29</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-30</title>
    <rule context="/xsd:schema">
      <assert test="fnx:is-exception('SD-30') or @version"
              flag="AUTO" role="ERROR">The version attribute is not specified.</assert>
      <assert test="fnx:is-exception('SD-30') or not(@version)
                    or matches(@version,'^V\d+_\d+$')" flag="AUTO" role="ERROR">
        &quot;<value-of select="@version"/>&quot; does not follow the
        form "V&lt;number&gt;_&lt;number&gt;"
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-31</title>
    <rule context="/">
      <assert test="fnx:is-exception('SD-31')
                    or $inSubdirName != 'Document'
                    or not(/xsd:schema/xsd:element)
                    or fnx:has-fixed-attribute(/xsd:schema/xsd:element, 'st96Version', 'required', 'V[0-9]+_[0-9]+')"
              flag="AUTO" role="ERROR">
        Document components must require root elements to have a st96Version attribute value of the form Vn_m.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-32</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-33</title>
    <rule context="xsd:enumeration">
      <assert test="fnx:is-exception('SD-33') or
                    not(matches(@value,'^other$', 'i'))" flag="AUTO" role="WARNING">The <value-of select="../../@name"/> enumeration should not include <value-of select="@value"/> value unless needed.</assert>
      <assert test="fnx:is-exception('SD-33') or
                    not(matches(@value,'^unknown$', 'i'))" flag="AUTO" role="WARNING">The <value-of select="../../@name"/> enumeration should not include <value-of select="@value"/> value unless needed.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-34</title>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'math')) or (@type and fnx:contains-i(@type, 'math'))]">
         <assert test="fnx:is-exception('SD-34') or
                    fnx:uses-type(., 'mathml:math', 0) or fnx:uses-type(., 'InlineFormula', 0)" flag="AUTO">
        Element <value-of select="@name"/>: mathml:math should be used for math data.
        <!-- WAS: Element <value-of select="@name"/>: mathml:math or
             InlineFormula should be used for math data. -->
         </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-35</title>
    <!-- check elements that are table-related but not table parts -->
    <rule context="xsd:element[fnx:isTableRelated(@name) or fnx:isTableRelated(@type)]">
      <assert test="fnx:is-exception('SD-35') or
                    fnx:uses-type(., 'tbl:OASISTableType', 0)" flag="AUTO" role="WARNING">
        Element <value-of select="@name"/> must use tbl:OASISTableType for table data.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-36</title>
    <rule context="xsd:import">
         <assert test="fnx:is-exception('SD-36') or starts-with(@namespace,$kNamespaceWIPObase) or 
                   @namespace = ($kMathMLNamespace,$kOASISTableNamespace)" flag="AUTO" role="WARNING">
        Import of namespace <value-of select="@namespace"/> does not appear to designate MathML, OASIS Table, or a WIPO or IPO namespace.
      </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-37</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-38</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-39</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
<!--
  places AcceptableException check on rule@context as opposed to on assert/@test:
  <pattern>
    <title>SD-40</title>
    <rule context="xsd:element[not(fnx:is-exception('SD-40')) and (@name and fnx:contains-i(@name, 'officecode')) or (@type and fnx:contains-i(@type, 'officecode'))]">
      <assert test="fnx:uses-type(., 'com:WIPOST3CodeType', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use com:WIPOST3CodeType for IP office codes.
      </assert>
    </rule>
  </pattern>
-->
  <pattern>
    <title>SD-40</title>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'officecode')) or (@type and fnx:contains-i(@type, 'officecode'))]">
      <assert test="fnx:is-exception('SD-40') or
                    fnx:is-exception('SD-40') or fnx:uses-type(., 'com:WIPOST3CodeType', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use com:WIPOST3CodeType for IP office codes.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-41</title>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'country')) or (@type and fnx:contains-i(@type, 'country'))]">
      <assert test="fnx:is-exception('SD-41') or
                    fnx:uses-type(., 'com:ISOCountryCodeType', 0)
                    or fnx:uses-type(., 'com:ExtendedWIPOST3CodeType', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use com:ISOCountryCodeType or com:ExtendedWIPOST3CodeType for country data.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-42</title>
    <rule context="xsd:element[(
                   @name
                   and fnx:contains-i(@name, 'language')
                   and not(ends-with(lower-case(@name), 'text'))
                   and not(ends-with(lower-case(@name), 'texttype'))
                   ) or (
                   @type
                   and fnx:contains-i(@type, 'language')
                   and not(ends-with(lower-case(@type), 'text'))
                   and not(ends-with(lower-case(@type), 'texttype'))
                   )]">
      <assert test="fnx:is-exception('SD-42') or
                    fnx:uses-type(., 'com:ISOLanguageCodeType', 0) or fnx:uses-type(., 'com:ExtendedISOLanguageCodeType', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use com:ISOLanguageCodeType or com:ExtendedISOLanguageCodeType for language codes.
      </assert>
    </rule>
  </pattern>
  <!--
  places AcceptableException check on rule@context as opposed to on assert/@test:
  <pattern>
    <title>SD-43</title>
    <rule context="xsd:element[not(fnx:is-exception('SD-43')) and (@name and fnx:contains-i(@name, 'datetime')) or (@type and fnx:contains-i(@type, 'datetime'))]">
      <assert test="fnx:uses-type(., 'xsd:dateTime', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:dateTime for date time data.
      </assert>
    </rule>
    <rule context="xsd:element[not(fnx:is-exception('SD-43')) and (@name and fnx:contains-i(@name, 'date')) or (@type and fnx:contains-i(@type, 'date'))]">
      <assert test="fnx:uses-type(., 'xsd:date', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:date for date data.
      </assert>
    </rule>
    <rule context="xsd:element[not(fnx:is-exception('SD-43')) and (@name and fnx:contains-i(@name, 'time')) or (@type and fnx:contains-i(@type, 'time'))]">
      <assert test="fnx:uses-type(., 'xsd:time', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:time for time data.
      </assert>
    </rule>
  </pattern>
  -->
  <pattern>
    <title>SD-43</title>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'datetime')) or (@type and fnx:contains-i(@type, 'datetime'))]">
      <assert test="fnx:is-exception('SD-43') or
                    fnx:is-exception('SD-43') or fnx:uses-type(., 'xsd:dateTime', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:dateTime for date time data.
      </assert>
    </rule>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'date')) or (@type and fnx:contains-i(@type, 'date'))]">
      <assert test="fnx:is-exception('SD-43') or
                    fnx:is-exception('SD-43') or fnx:uses-type(., 'xsd:date', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:date for date data.
      </assert>
    </rule>
    <rule context="xsd:element[(@name and fnx:contains-i(@name, 'time')) or (@type and fnx:contains-i(@type, 'time'))]">
      <assert test="fnx:is-exception('SD-43') or
                    fnx:is-exception('SD-43') or fnx:uses-type(., 'xsd:time', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use xsd:time for time data.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-44</title>
    <rule context="xsd:element[
                      (@name and (fnx:contains-i(@name, 'currency') or fnx:contains-i(@name, 'money')))
                   or (@type and (fnx:contains-i(@type, 'currency') or fnx:contains-i(@type, 'money')))]">
      <assert test="fnx:is-exception('SD-44') or
                    fnx:uses-type(., 'com:ISOCurrencyCodeType', 0)" flag="AUTO" role="ERROR">
        Element <value-of select="@name"/> must use com:ISOCurrencyCodeType for currency codes.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-45</title>
    <rule context="xsd:enumeration">
      <assert test="fnx:is-exception('SD-45') or
                    matches(@value,'^[a-zA-Z0-9_ ]+$')" flag="AUTO" role="WARNING">The enumeration value <value-of select="@value"/> has invalid characters.</assert>
      <assert test="fnx:is-exception('SD-45') or
                    not(matches(@value,'^[0-9]'))" flag="AUTO" role="WARNING">The enumeration value <value-of select="@value"/> starts with a numeric character.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-46</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-47</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-48</title>
    <rule context="xsd:attribute[@name]">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-49</title>
    <rule context="xs:element">
         <assert test="fnx:is-exception('SD-49') or
                    not(@minOccurs = '1')" role="WARNING">
        @minOccurs is set to the default value (1) for <value-of select="name()"/> (<value-of select="@name|@ref"/>).
      </assert>
         <assert test="fnx:is-exception('SD-49') or
                    not(@maxOccurs = '1')" role="WARNING">
        @maxOccurs is set to the default value (1) for <value-of select="name()"/> (<value-of select="@name|@ref"/>).
      </assert>
      </rule>
      <rule context="xs:sequence | xs:any | xs:all | xs:choice | xs:group ">
         <assert test="fnx:is-exception('SD-49') or
                    not(@minOccurs = '1')" role="WARNING">
        @minOccurs is set to the default value (1) for <value-of select="name()"/>.
      </assert>
         <assert test="fnx:is-exception('SD-49') or
                    not(@maxOccurs = '1')" role="WARNING">
        @maxOccurs is set to the default value (1) for <value-of select="name()"/>.
      </assert>
      </rule>
  </pattern>
  <pattern>
    <title>SD-50</title>
    <rule context= "xs:element">
      <assert test="fnx:is-exception('SD-50') or 
                    @name = 'Br'
                    or (    not(xsd:simpleType/xsd:restriction[@base='xsd:token' or @base='xsd:string']/xsd:maxLength[@value=0])
                    and not(xsd:complexType/xsd:complexContent/xsd:restriction[@base='xsd:anyType' and not(*)])
                    )" role="ERROR">
        The <value-of select="@name"/> element is empty.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-51</title>
    <rule context="xsd:attribute">
      <assert test="fnx:is-exception('SD-51') or
                    not(@use = 'optional')" flag="AUTO"
      role="WARNING">Attribute <value-of select="@ref|@name"/> has a
      use of "optional", the default value.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-52</title>
    <rule context="xsd:complexType">
      <assert test="fnx:is-exception('SD-52') or
                    not(.//xsd:all)" flag="AUTO"
      role="WARNING">&quot;<value-of
      select="ancestor-or-self::*/@name[1]"/>&quot; uses
      xsd:all.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-53</title>
    <rule context="xsd:sequence[@minOccurs | @maxOccurs] | xsd:choice[@minOccurs | @maxOccurs]">
      <!--
          TODO: may want to be manual but check what action we can get
          out of role="PASS" with Oxygen.

          DONE: Oxygen by default does not recognize role="PASS",
          just: warning/warn, fatal information/info, and error.
      -->
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-54</title>
    <rule context="xsd:complexType">
      <assert test="fnx:is-exception('SD-54') or
                    not(.//xsd:any)" flag="AUTO"
      role="ERROR">&quot;<value-of
      select="ancestor-or-self::*/@name[1]"/>&quot; uses
      xsd:any.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-55</title>
    <rule context="xsd:element">
      <assert test="fnx:is-exception('SD-55') or
                    not(@substitutionGroup)" flag="AUTO"
      role="ERROR">Element <value-of select="@name"/> uses
      substitution groups.</assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-56</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-57</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-58</title>
    <rule context="xsd:element[@name]|xsd:attribute[@name]">
         <assert test="fnx:is-exception('SD-58') or
                    xsd:annotation/xsd:documentation" flag="AUTO" role="WARNING">
        The <value-of select="concat(@name, ' ', local-name(.))"/>
        is missing xsd:documentation.
      </assert>
         <report test="false()" flag="MANUAL"/>
      </rule>
  </pattern>
  <pattern>
    <title>SD-59</title>
    <rule context="/">
      <assert test="fnx:is-exception('SD-59') or
                    not(comment())" flag="AUTO" role="WARNING">
        Outside of the root element, there are XML comments containing
        the following text: <value-of select="comment()"/>
      </assert>
    </rule>
    <rule context="*">
      <assert test="fnx:is-exception('SD-59') or
                    not(./comment())" flag="AUTO" role="WARNING">
        Element (<value-of select="name(.)"/>) has XML comments
        containing the following text: <value-of select="comment()"/>
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-60</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-61</title>
    <rule context="/xsd:schema">
	  <assert test="fnx:is-exception('SD-61')
                    or $inSubdirName != 'Document'
                    or count(/xsd:schema/xsd:complexType) &gt; 0
                    or (count (/xsd:schema/xsd:annotation/xsd:appinfo) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo) &gt; 0)"
              flag="AUTO" role="ERROR">
        The xsd:schema does not contain an xsd:annotation/xsd:documentation element.
      </assert>
	  <assert test="fnx:is-exception('SD-61')
                    or $inSubdirName != 'Document'
                    or count(/xsd:schema/xsd:complexType) &gt; 0
                    or not(count (/xsd:schema/xsd:annotation/xsd:appinfo) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo) &gt; 0)
                    or (count (/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaCreatedDate) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaCreatedDate) &gt; 0)"
              flag="AUTO" role="ERROR">
        The xsd:annotation/xsd:appinfo element does not contain the required &quot;com:SchemaCreatedDate&quot; element.
      </assert>
	  <assert test="fnx:is-exception('SD-61')
                    or $inSubdirName != 'Document'
                    or count(/xsd:schema/xsd:complexType) &gt; 0
                    or not(count (/xsd:schema/xsd:annotation/xsd:appinfo) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo) &gt; 0)
                    or (count (/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaLastModifiedDate) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaLastModifiedDate) &gt; 0)"
              flag="AUTO" role="ERROR">
        The xsd:annotation/xsd:appinfo element does not contain the required &quot;com:SchemaLastModifiedDate&quot; element.
      </assert>
	  <assert test="fnx:is-exception('SD-61')
                    or $inSubdirName != 'Document'
                    or count(/xsd:schema/xsd:complexType) &gt; 0
                    or not(count (/xsd:schema/xsd:annotation/xsd:appinfo) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo) &gt; 0)
                    or (count (/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaContactPoint) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaContactPoint) &gt; 0)"
              flag="AUTO" role="ERROR">
        The xsd:annotation/xsd:appinfo element does not contain the required &quot;com:SchemaContactPoint&quot; element.
      </assert>
	  <assert test="fnx:is-exception('SD-61')
                    or $inSubdirName != 'Document'
                    or count(/xsd:schema/xsd:complexType) &gt; 0
                    or not(count (/xsd:schema/xsd:annotation/xsd:appinfo) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo) &gt; 0)
                    or (count (/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaReleaseNoteURL) = 1
							and string-length(/xsd:schema/xsd:annotation/xsd:appinfo/com:SchemaReleaseNoteURL) &gt; 0)"
              flag="AUTO" role="WARNING">
        The xsd:annotation/xsd:appinfo element does not contain the optional &quot;com:SchemaReleaseNoteURL&quot; element.
      </assert>
    </rule>
  </pattern>
  <pattern>
    <title>SD-62</title>
    <rule context="/">
      <report test="false()" flag="MANUAL"/>
    </rule>
  </pattern>
  <pattern>
    <title>SD-63</title>
    <rule context="/">
      <assert test="fnx:is-exception('SD-63')
                    or $inSubdirName != 'Document'
                    or not(/xsd:schema/xsd:element)
                    or fnx:has-fixed-attribute(/xsd:schema/xsd:element, 'ipoVersion', 'optional', '[A-Z][A-Z]_V[0-9]+_[0-9]+')"
              flag="AUTO" role="ERROR">
        Root element does not declare an optional ipoVersion attribute
        with a fixed value of the form {ST3CODE}_V{n}_{m}.
      </assert>
    </rule>
    <rule context="xsd:attribute[@fixed]">
      <assert test="fnx:is-exception('SD-63')
                    or not(matches(@fixed, '[A-Z][A-Z]_V[0-9]+_[0-9]+'))
                    or fnx:starts-with-ST3-code(@fixed)"
              flag="AUTO" role="ERROR">
        @ipoVersion=<value-of select="@fixed"/> is fixed but does not start with a valid ST.3 code.
      </assert>
    </rule>
  </pattern>

  <pattern>
<!--
OLD: none
NEW: Schema release folder, a document level schema filename and a flattened schema filename MUST contain matching version information comprising the major and minor version number.
-->
    <title>SD-64</title>
    <let name="releaseDirName" value="if (matches(base-uri(.),$kRegexReleaseDir)) then replace(base-uri(.), $kRegexReleaseDir, '$1') else ''"/>
    <let name="releaseMajorVersionNum"
         value="if (matches($releaseDirName,$kRegexReleaseVersion)) then replace($releaseDirName,$kRegexReleaseVersion,'$1') else ''"/>
    <let name="releaseMinorVersionNum"
         value="if (matches($releaseDirName,$kRegexReleaseVersion)) then replace($releaseDirName,$kRegexReleaseVersion,'$2') else ''"/>
    <let name="fileMajorVersionNum"
         value="if (matches($localFile,$kRegexLocalFileVersion)) then replace($localFile,$kRegexLocalFileVersion,'$1') else ''"/>
    <let name="fileMinorVersionNum"
         value="if (matches($localFile,$kRegexLocalFileVersion)) then replace($localFile,$kRegexLocalFileVersion,'$2') else ''"/>
    <rule context="/">
      <assert test="fnx:is-exception('SD-64')
                    or not($isFlattened)
                    or matches($localFile, $kRegexLocalFileVersion)" flag="AUTO">
        The flattened schema file name <value-of select="$localFile"/> does not contain version information in the required format &#x003C;ComponentName&#x003E;[_V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E;].&#x003C;FileExtension&#x003E;
      </assert>
      <assert test="fnx:is-exception('SD-64')
                    or not($isFlattened)
                    or $releaseDirName != ''" flag="AUTO">
        No schema release folder was found with the required format V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E; for flattened XSD named <value-of select="$localFile"/>.
      </assert>
      <assert test="fnx:is-exception('SD-64')
                    or $inSubdirName != 'Document'
                    or matches($localFile, $kRegexLocalFileVersion)" flag="AUTO">
        The Document schema file name <value-of select="$localFile"/> does not contain version information in the required format &#x003C;ComponentName&#x003E;[_V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E;].&#x003C;FileExtension&#x003E;
      </assert>

      <assert test="fnx:is-exception('SD-64')
                    or $inSubdirName != 'Document'
                    or ($releaseDirName != ''
                        and $releaseMajorVersionNum != ''
                        and $releaseMinorVersionNum != '')" flag="AUTO">
        No schema release folder was found with the required format ST96/V&#x003C;MajorVersionNum&#x003E;_&#x003C;MinorVersionNum&#x003E; for Document XSD named <value-of select="$localFile"/>.
      </assert>

      <assert test="fnx:is-exception('SD-64')
                    or not($isFlattened)
                    or $releaseMajorVersionNum = ''
                    or $releaseMinorVersionNum = ''
                    or $fileMajorVersionNum = ''
                    or $fileMinorVersionNum = ''
                    or ($releaseMajorVersionNum = $fileMajorVersionNum)
                    " flag="AUTO">
        The flattened schema file name <value-of select="$localFile"/> has a major version number (<value-of select="$fileMajorVersionNum"/>) that does not match the release major version number (<value-of select="$releaseMajorVersionNum"/>).
      </assert>
      <assert test="fnx:is-exception('SD-64')
                    or not($isFlattened)
                    or $releaseMajorVersionNum = ''
                    or $releaseMinorVersionNum = ''
                    or $fileMajorVersionNum = ''
                    or $fileMinorVersionNum = ''
                    or ($releaseMinorVersionNum = $fileMinorVersionNum)
                    " flag="AUTO">
        The flattened schema file name <value-of select="$localFile"/> has a minor version number (<value-of select="$fileMinorVersionNum"/>) that does not match the release minor version number (<value-of select="$releaseMinorVersionNum"/>).
      </assert>

      <assert test="fnx:is-exception('SD-64')
                    or $inSubdirName != 'Document'
                    or $releaseMajorVersionNum = ''
                    or $releaseMinorVersionNum = ''
                    or $fileMajorVersionNum = ''
                    or $fileMinorVersionNum = ''
                    or ($releaseMajorVersionNum = $fileMajorVersionNum)
                    " flag="AUTO">
        The Document schema file name <value-of select="$localFile"/> has a major version number (<value-of select="$fileMajorVersionNum"/>) that does not match the release major version number (<value-of select="$releaseMajorVersionNum"/>).
      </assert>
      <assert test="fnx:is-exception('SD-64')
                    or $inSubdirName != 'Document'
                    or $releaseMajorVersionNum = ''
                    or $releaseMinorVersionNum = ''
                    or $fileMajorVersionNum = ''
                    or $fileMinorVersionNum = ''
                    or ($releaseMinorVersionNum = $fileMinorVersionNum)
                    " flag="AUTO">
        The Document schema file name <value-of select="$localFile"/> has a minor version number (<value-of select="$fileMinorVersionNum"/>) that does not match the release minor version number (<value-of select="$releaseMinorVersionNum"/>).
      </assert>

      <!--
      <assert test="false()" flag="AUTO">
        base-uri(.)=<value-of select="base-uri(.)"/>
        inSubdirName=<value-of select="$inSubdirName"/>
        localFile=<value-of select="$localFile"/>
        releaseDirName=<value-of select="$releaseDirName"/>
        releaseMajorVersionNum=<value-of select="$releaseMajorVersionNum"/>
        releaseMinorVersionNum=<value-of select="$releaseMinorVersionNum"/>
        fileMajorVersionNum=<value-of select="$fileMajorVersionNum"/>
        fileMinorVersionNum=<value-of select="$fileMinorVersionNum"/>
      </assert>
      -->

    </rule>

  </pattern>


</schema>
